/*----------------------------------------------------------------------------------+
|                                                                                   |
|                    MITEL Integrated DTMF Transceiver (MT8880C)                    |
|                                                                                   |
| Name: MT8880C.c                                                                   |
| Purpose:                                                                          |
|	DTMF stands for dual tone multiple frequency. When you press a button on a      |
|	telephone keypad, the telephone will generate two sine waves with different     |
|	frequency at the same time, the audible sounds that you hear. This program      |
|   is used for the access and control the DTMF Transceiver, to transmit tone, or   |
|   recevice tone from telephone line. software reset must be implement, before any |
|   transceive function.                                                            |
|                                                                                   |
| By Dillian Wong, last modify data 27/1/04                                         |
+----------------------------------------------------------------------------------*/

/*____________ I N C L U D E S ____________________________________________________*/

#include <reg51.h>
#include <stdio.h>
#include "configMT8880C.h"

/*____________ M A C R O ____________________________________________________*/

/* set or clr of important PIN */
#define setCS CS=1;while (CS!=1); // chip select, clr for select
#define clrCS CS=0;
#define setRS RS=1;while (RS!=1); // register select, set for select
#define clrRS RS=0;
#define setRW RW=1;while (RW!=1); // read write, set for read, clr for write
#define clrRW RW=0;

/*____________ P R O T O C O L ____________________________________________________*/

void MT8880C_Reset(void);						// RESET
void MT8880C_SetForTransmit_tone(void);			// TRANSMIT
void MT8880C_TransDTMF_tone(char ch);
char MT8880C_ReceiveDTMF_tone(void);			// RECEIVE
void MT8880C_ReadFromStatusRegister(void);
void MT8880C_ReadFromReceiveDataRegister(void);
int digit2ToneNo(char ch);						// CONVERTION
char toneNo2Digit(int num);


/*____________ DTMF transceiver iitialization______________________________________*/

/*
	A software reset must be included at the beginning of all programs to initialize the 
	control registers after power up. The initialization procedure should be implemented
	100ms after power up.
*/
void MT8880C_Reset(void)
{
	int i;

	setCS;		//1) read status register
	setRS;setRW;
	clrCS;
	for(i=0;i<3;i++);
	setCS;

	setCS;		//2) write to control register
	setRS;clrRW;b3=0;b2=0;b1=0;b0=0;
	clrCS;
	for(i=0;i<3;i++);
	setCS;

	setCS;		//3) write to control register
	setRS;clrRW;b3=0;b2=0;b1=0;b0=0;
	clrCS;
	for(i=0;i<3;i++);
	setCS;

	setCS;		//4) write to control register
	setRS;clrRW;b3=1;b2=0;b1=0;b0=0;
	clrCS;
	for(i=0;i<3;i++);
	setCS;

	setCS;		//5) write to control register
	setRS;clrRW;b3=0;b2=0;b1=0;b0=0;
	clrCS;
	for(i=0;i<3;i++);
	setCS;

	setCS;		//6) read status register
	setRS;setRW;
	clrCS;
	for(i=0;i<3;i++);
	setCS;
}


/*____________ DTMF tone transmitter function_________________________________________*/

/*
	setting before transmit a tone.
*/
void MT8880C_SetForTransmit_tone(void)
{
	int m;

	/*tone out, DTMF, IRQ, select control regiser B*/
	setCS;					//write to control register A
	setRS;	clrRW;	b3=1;	b2=1;	b1=0;	b0=1;
	clrCS;					//toggle chip select to fatch the input data
	for(m=0;m<3;m++);
	setCS;
	
	/*burst mode*/
	setCS;					//write to control register B
	setRS;	clrRW;	b3=0;	b2=0;	b1=0;	b0=0;
	clrCS;					//toggle chip select to fatch the input data
	for(m=0;m<3;m++);
	setCS;
}

/*
	Transmit a DTMF tone, ch is the tone digit (0-9,*,#,A-D)
*/
void MT8880C_TransDTMF_tone(char ch)
{
	int m;
	int i=0,j=0,k=0,l=0,digit;

	digit=digit2ToneNo(ch);	//get the interger that represent the tone

	i=(digit&0x0008)/8;		//calculate the b3,b2,b1,b0
	j=(digit&0x0004)/4;
	k=(digit&0x0002)/2;
	l=(digit&0x0001)/1;

	setCS;					//write to transmit data register
	clrRS;	clrRW;	b3=i;	b2=j;	b1=k;	b0=l;
	clrCS;					//toggle chip select to fatch the input data
	for(m=0;m<3;m++);
	setCS;
}


/*____________ DTMF tone receiver function___________________________________________*/

/*
	if DTMF tone received, return the received DTMF tones, else return 'N' indicate
	no DTMF tone received.
*/
char MT8880C_ReceiveDTMF_tone(void)
{
	int i=0,j=0,k=0,l=0,num=0;
	char ch;

	MT8880C_ReadFromStatusRegister();			//code for read status
	if(b2==1)									//if bit2 is set, a DTMF tone has been
	{											//received, in which case....
		MT8880C_ReadFromReceiveDataRegister();	//code for read data
		i=b3;j=b2;k=b1;l=b0;
		num=i*8+j*4+k*2+l;						//convert 4 binary to integer
		ch=toneNo2Digit(num);					//get the relate digit character
		return ch;
	}
	else
		return 'N';
}

/*
	read the status register
*/
void MT8880C_ReadFromStatusRegister(void)
{
	int i=0;
	/*Read the register*/
	setCS;					//disable chip select			
	setRS;	setRW;			//select register, select read
	setCS;					//toggle chip select to fatch the input data
	for(i=0;i<30;i++);
	clrCS;					//chip select
}

/*
	read the received data register
*/
void MT8880C_ReadFromReceiveDataRegister(void)
{
	int i=0;
	/*Read the register*/
	setCS;					
	clrRS;	setRW;			//not register, select read
	setCS;					//toggle chip select to fatch the input data
	for(i=0;i<30;i++);
	clrCS;
}


/*____________ tone Number & DTMF digit Convertion___________________________________*/

/*
	return the interger that represent the digal number or character.
	1-9,0,*,#,A-D.
*/
int digit2ToneNo(char ch)
{
	int digit;
	switch(ch)
	{	//digit     toneNo
		case '1':	digit=1;	break;
		case '2':	digit=2;	break;
		case '3':	digit=3;	break;
		case '4':	digit=4;	break;
		case '5':	digit=5;	break;
		case '6':	digit=6;	break;
		case '7':	digit=7;	break;
		case '8':	digit=8;	break;
		case '9':	digit=9;	break;
		case '0':	digit=10;	break;
		case '*':	digit=11;	break;
		case '#':	digit=12;	break;
		case 'A':	digit=13;	break;
		case 'B':	digit=14;	break;
		case 'C':	digit=15;	break;
		case 'D':	digit=0;	break;
		default:	digit=0;	break;
	}
	return digit;
}

/*
	convert the tone number that represent the digit.
*/
char toneNo2Digit(int num)
{
	char ch;
	switch(num)
	{	//toneNo	digit
		case 1:		ch='1';		break;
		case 2:		ch='2';		break;
		case 3:		ch='3';		break;
		case 4:		ch='4';		break;
		case 5:		ch='5';		break;
		case 6:		ch='6';		break;
		case 7:		ch='7';		break;
		case 8:		ch='8';		break;
		case 9:		ch='9';		break;
		case 10:	ch='0';		break;
		case 11:	ch='*';		break;
		case 12:	ch='#';		break;
		case 13:	ch='A';		break;
		case 14:	ch='B';		break;
		case 15:	ch='C';		break;
		case 0:		ch='D';		break;
	}
	return ch;
}
