/*----------------------------------------------------------------------------------+
|                                                                                   |
|                    Ditital Thermometer and Thermostat (DS1620)                    |
|                                                                                   |
| Name: DS1620.c                                                                    |
| Purpose:                                                                          |
|                                                                                   |
| By Dillian Wong, last modify data 4/10/03                                         |
+----------------------------------------------------------------------------------*/

/*____________ I N C L U D E S ____________________________________________________*/

#include <reg51.h>
#include <stdio.h>
#include "rPrintf\rprintf.h"

/*____________ M A C R O S ________________________________________________________*/

#define READ_TEMPERATURE 	0xAA		// reads last converted temperature value
#define START_CONVERT_T 	0xEE		// initates temperature conversion
#define STOP CONVERT_T  	0x22		// halts temperature conversion
#define WRITE_TH 			0x01		// write high temperature limit
#define WRITE_TL 			0x02		// write low temperature limit
#define READ_TH				0xA1		// read high temperature limit
#define READ_TL 			0xA2		// read low temperature limit
#define WRITE_CONFIG 		0x0C		// write configuration data
#define READ_CONFIG 		0xAC		// read configuration data
#define CONTINUOUS			0x02		// set for continuous conversion
#define	ONESHOT				0x01		// set for one temperature conversion
#define TEMPSTEP 			0.5			// digit step of the temperature
#define TOOLOW				0x00
#define GOOD				0x01
#define TOOHIGH				0x02

/*____________ H A R D W A R E ____________________________________________________*/

sbit DQ     = P3^5;
sbit CLK	= P3^6;
sbit RST	= P3^7;
/* set or clr of important PIN */
#define setDQ		DQ=1;while (DQ!=1);						// data in 
#define clrDQ		DQ=0;
#define setCLK		CLK=1;while (CLK!=1);					// clock
#define clrCLK		CLK=0;
#define setRST		RST=1;while (RST!=1);					// chip reset
#define clrRST		RST=0;

/*____________ P R O T O C O L ____________________________________________________*/

void ds1620_init(float TH, float TL);
void  ds1620_sendByte(unsigned char dat);
unsigned char ds1620_readByte(void);
float ds1620_toDegree(unsigned char dat);
float ds1620_ReadTemperature(void);
float ds1620_ReadTH(void);
float ds1620_ReadTL(void);

/*____________ F U N C T I O N ____________________________________________________*/

/*
	initalize the digital thermometer with the High temperature trigger and Low
	temperature trigger. 
*/
void ds1620_init(float TH, float TL)	
{
	int numTH,numTL;					// no. step
	unsigned char thH,thL,tlH,tlL;		

	/* convert the float input to digit step */
	numTH=TH/TEMPSTEP;					// the no. step of TH
	numTL=TL/TEMPSTEP;					// the no. step of TL

	// high temperature limit
	thH=((numTH&0xff00)/0x00ff);		// high byte of TH
	if(!(TH>0))
	{
		thH = 0x01;
	}	
	thL=((numTH&0x00ff));				// low byte of TH

	// low temperature limit
	tlH=((numTL&0xff00)/0x00ff);		// high byte of TH
	if(!(TL>0))
	{
		tlH = 0x01;
	}	
	tlL=((numTL&0x00ff));				// low byte of TH

/* 
	printf("Digital Code is : ");
	printHex8(thH);
	printHex8(thL);
	printHex8(tlH);
	printHex8(tlL);
*/
	/* start the configuration */
	ds1620_sendByte(WRITE_CONFIG);
	ds1620_sendByte(CONTINUOUS);			
	clrRST;
	setRST;
	ds1620_sendByte(WRITE_TH);			// set TH
	ds1620_sendByte(thL);
	ds1620_sendByte(thH);
	clrRST;
	setRST;
	ds1620_sendByte(WRITE_TL);			// set TL
	ds1620_sendByte(tlL);
	ds1620_sendByte(tlH);
	clrRST;
	setRST;
	ds1620_sendByte(START_CONVERT_T);			// issues start convert T command

}


/*
	send 8 bit data to the digital thermometer.
*/
void  ds1620_sendByte(unsigned char dat)
{
	unsigned char i,mask=1;
	setRST;
	for (i=0; i<8; i++)
	{
		clrCLK;						
		DQ = (dat & mask);						// send bit to 1620
		setCLK;
		mask=(mask<<1);   				 		// setup to send next bit
	}
}

/*
	get 8 bit data from the digital thermometer.
*/
unsigned char ds1620_readByte(void)
{
	unsigned char i,rcv=0,mask=1;
	for (i=0; i<8; i++)
	{
		clrCLK;
		if (DQ) rcv|=mask; 						// read bit and or if = 1
		setCLK;
		mask=(mask<<1); 						// setup for next read and or 
	}
	return rcv;
}

/*
	the binary data represend a interger step. For 0x01, it may represent 5, for 0x03,
	it may represnet 15. So. this function is for the conversation of the binary data
	to the refered interger.
*/
float ds1620_toDegree(unsigned char dat)
{
	float fdeg=0;
	unsigned int i;

	// only for positive data, if dat is negative, the 2 complement will required.
	for(i=0;i<dat;i++)
	{
		fdeg +=0.5;
	}	

	return fdeg;
}

/*
	issues Ds1620 measures the current temperature and then read data from the 
	thermometer,convert digit ouputs and return the decimal interger of temperature.
*/
float ds1620_ReadTemperature(void)
{
	unsigned char datH,datL;
	float deg;

	clrRST;
	setRST;
	ds1620_sendByte(READ_TEMPERATURE);			// read temperature T
	datL = ds1620_readByte();
	datH = ds1620_readByte();

	if(datH&0x01)
	{
		deg = -ds1620_toDegree(~datL+1 );		// if the temperature is negative
	}											// take complement
	else
	{
		deg = ds1620_toDegree(datL);			// if the temperature is positive
	}	

	return deg;									// return T in interger form
}

/*
	reads stored valuse of high temperature limit from TH register.
*/
float ds1620_ReadTH()
{
	unsigned char datH,datL;
	float deg;

	clrRST;
	setRST;
	ds1620_sendByte(READ_TH);				// read TH
	datL = ds1620_readByte();
	datH = ds1620_readByte();

	if(datH&0x01)
	{
		deg = -ds1620_toDegree(~datL+1 );		// if the temperature is negative
	}											// take complement
	else
	{
		deg = ds1620_toDegree(datL);			// if the temperature is positive
	}	

	return deg;									// return T in interger form

}

/*
	reads stored valuse of low temperature limit from TL register.
*/
float ds1620_ReadTL()
{
	unsigned char datH,datL;
	float deg;

	clrRST;
	setRST;
	ds1620_sendByte(READ_TL);				// read TH
	datL = ds1620_readByte();
	datH = ds1620_readByte();

	if(datH&0x01)
	{
		deg = -ds1620_toDegree(~datL+1 );		// if the temperature is negative
	}											// take complement
	else
	{
		deg = ds1620_toDegree(datL);			// if the temperature is positive
	}	

	return deg;									// return T in interger form

}


