/*-----------------------------------------------------------------------------------------------|
|                                                                                                |
|                                Eeprom Accesser (AT24C16A)                                      |
|                                                                                                |
|  Name   : EepromAccess.c                                                                       |
|  Purpose:                                                                                      |
|      Provide a interface for easier access the operation of the eeprom AT24C16A. Via these     |
|      function, user can easily access the memory storage device. save/load a character; save/  |
|      load a integer from Eeprom, etc                                                           |
|-----------------------------------------------------------------------------------------------*/

/*____________ I N C L U D E S _________________________________________________________________*/


#include <reg51.h>
#include <stdio.h>
#include "Eepromconfig.h"

/*____________ P R O T O C O L _________________________________________________________________*/

void saveChar(unsigned char addr,unsigned char ch);
unsigned char loadChar(unsigned char addr);
void saveInt(unsigned char addr, int intData);
int loadInt(unsigned char addr);


/*_____FUNCTION USED FOR EASILY ASSESS THE MEMORY STORAGE AND LOADING FORM I2C DEVICE___________*/


/*
	save a char to the Eeprom in address specificed by variable addr.
*/
void saveChar(unsigned char addr,unsigned char ch)
{
	write_Byte(addr, ch);
}

/*	
	load a char from address indicated by variable addr.
*/
unsigned char loadChar(unsigned char addr)
{
 	return read_Byte(addr);
}

/*
	save a integer to Eeprom, use 2 bytes of data from address indicated by variable
	addr. the intData will be divided into High byte and Low byte, then stored into 2
	bytes.
*/
void saveInt(unsigned char addr, int intData)
{
	unsigned char H,L,sign=0x00;
	if(intData<0){									// a char used to store
		intData=-intData;							// the sign of intData
		sign = 0x01;
	}
	H=((intData&0xff00)/0x00ff);					// high byte 
	L=((intData&0x00ff));							// low byte 
	write_Byte(addr,sign);							// data store
	write_Byte(addr+1,L);								
	write_Byte(addr+2,H);
}

/*
	load a integer from addr. Get 2 bytes of data then combine it to form a integer.
*/
int loadInt(unsigned char addr)
{
	unsigned int intData=0x0000;					// initalize avoid destruction
	unsigned char H,L,T,sign;						// must unsinged for char & int, as H, L must a 
													// positve value for intData. (extra byte is used
													// to indicate the sign of intData).
	sign=read_Byte(addr);							// extract data
	L=read_Byte(addr+1);
	H=read_Byte(addr+2);

	intData = (H*256) + L;							// combine H byte & L byte

	/*
		Bug elimination, reason still not know. Maybe due to overflow of calculation. When intData
		>=0x0100, it will less 256. eg. 0x3489-->0x3389, 0xedx66-->0xec66. 0x0045-->0x0045. Thus
		it is need to add 256 to intData, when intData>=0x0100.
	*/
	T=((intData&0xff00)/0x00ff);					// extract combined H byte
	if(T<H)											// when intData > =0x0100, add 256 to it.
		intData=(H*256) + L + 256;					

	if(sign==0x00)									// 0x00 means positive							
		return intData;					
	else											// 0x01 means negative
		return -intData;
}
