#include "../Socket.h"
#include "../commandCodes.h"
#include "main.h"
#include "resource.h"
#include <fstream>
#include <math.h>
#include <commctrl.h>

//Socket shutdown checker
inline bool SERR(int res)
{return (res == 0 || res == SOCKET_ERROR);}

//Protocol defines
#define FILESEND_PACKETSIZE 4096

//dlgState defines
#define DS_CONNECT 1
#define DS_LOGIN 2
#define DS_FILES 3
#define DS_ADMIN 4
#define DS_ADMINFILES 5
#define DS_USERS 6

//Global Variables
bool quit = false;
int dlgState = DS_CONNECT;
unsigned char folder = ICV_F_PERSONAL;
bool guest = false;

HINSTANCE hinst;
Socket* server;
char hostbuf[256];

//+----------------------+
//|     *WinMain()*      |
//+----------------------+

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR cmd, int showCmd)
{
	hinst = hInst;

	appInit();

	while(!quit)
	{
		switch(dlgState)
		{
		case DS_CONNECT:
			dlgConnect();
			break;
		case DS_LOGIN:
			dlgLogin();
			break;
		case DS_FILES:
			dlgFiles();
			break;
		case DS_ADMIN:
			dlgAdmin();
			break;
		case DS_ADMINFILES:
			dlgAdminFiles();
			break;
		case DS_USERS:
			dlgUsers();
			break;
		}
	}

	appShutdown();

	return 0;
}


//+-----------------------+
//|   *Dialog Startups*   |
//+-----------------------+

inline void dlgConnect()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_CONNECT), NULL, ConnectProc);
}
inline void dlgLogin()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_LOGIN), NULL, LoginProc);
}
inline void dlgFiles()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_FILES), NULL, FilesProc);
}
inline void dlgSetShare(HWND parent)
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_SHARE), parent, ShareProc);
}
inline void dlgAdmin()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_REMOTEADMIN), NULL, AdminProc);
}
inline void dlgAdminFiles()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_FILES), NULL, AdminFilesProc);
}
inline void dlgUsers()
{
	DialogBox(hinst, MAKEINTRESOURCE(IDD_USERINFO), NULL, UserInfoProc);
}


//+--------------------------+
//|   *Callback Functions*   |
//+--------------------------+

BOOL CALLBACK ConnectProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	std::ifstream inFile;
	std::ofstream outFile;
	switch(msg)
	{
	case WM_INITDIALOG:
		inFile.open("iphistory.dat");
		if(!inFile)
			return false;

		char tempbuf[15];
		HWND ctrl;
		ctrl = GetDlgItem(hWnd, IDC_ADDR);
		while(true)
		{
			inFile.getline(tempbuf, 15);
			if(inFile.eof())
				break;
			SendMessage(ctrl, CB_ADDSTRING, 0, (LPARAM)tempbuf);
		}
		inFile.close();
		return false;

	case WM_CLOSE:
		quit = true;
		EndDialog(hWnd, 0);
		return true;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_CONNECT:
			GetDlgItemText(hWnd, IDC_ADDR, hostbuf, 256);
			if(strcmp(hostbuf, "") == 0)
			{
				MessageBox(hWnd, "Please enter a server address.", "Error", MB_OK);
				break;
			}
			if(!server->connect(hostbuf, ICV_PORT))
			{
				MessageBox(hWnd, "Unable to connect to server!", "Error", MB_OK);
				server->release();
				delete server;
				server = new Socket;
				break;
			}
			outFile.open("icvtemp.tmp");
			outFile << hostbuf << "\n";
			inFile.open("iphistory.dat");
			if(inFile.is_open())
			{
				char tempbuf[15];
				for(int i = 0; i < 5; ++i)
				{
					inFile.getline(tempbuf, 15);
					if(inFile.eof())
						break;
					outFile << tempbuf << "\n";
				}
				inFile.close();
			}
			outFile.close();
			DeleteFile("iphistory.dat");
			rename("icvtemp.tmp", "iphistory.dat");

			dlgState = DS_LOGIN;
			EndDialog(hWnd, 0);
			break;
		case IDC_QUIT:
			quit = true;
			EndDialog(hWnd,0);
			break;
		}
		return true;
	}
	return false;
}
BOOL CALLBACK LoginProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	if(serverShutdown())
	{
		//MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		//backToConnect(hWnd);
		
		server->release();
		delete server;
		server = new Socket;
		MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
	}

	std::ifstream inFile;
	std::ofstream outFile;

	HWND ctrl;
	switch(msg)
	{
	case WM_INITDIALOG:
		SetWindowLong(hWnd, GWL_USERDATA, 2);
		CheckRadioButton(hWnd, IDC_NORMAL, IDC_NORMAL, IDC_NORMAL);
		inFile.open("userhistory.dat");
		if(!inFile)
			return false;

		char temp[256];
		while(true)
		{
			inFile.getline(temp, 256);
			if(inFile.eof())
				break;
			SendDlgItemMessage(hWnd, IDC_USERNAME, CB_ADDSTRING, 0, (LPARAM)temp);
		}
		return false;

	case WM_CLOSE:
		quit = true;
		server->send(htonl(1));
		server->send(ICV_OP_DISCONNECT);
		EndDialog(hWnd, 0);
		return true;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_LOGIN:
			char nameBuf[256];
			unsigned char nameLen;
			char pass[256];
			unsigned char passLen;

			if(GetWindowLong(hWnd, GWL_USERDATA) == 2) //Normal
			{
				GetDlgItemText(hWnd, IDC_USERNAME, nameBuf, 255);
				nameLen = strlen(nameBuf);
				if(nameLen == 0)
				{
					MessageBox(hWnd, "Please enter a username.", "Error", MB_OK);
					break;
				}
				GetDlgItemText(hWnd, IDC_PASSWORD, pass, 255);
				passLen = strlen(pass);
				if(passLen == 0)
				{
					MessageBox(hWnd, "Please enter a password.", "Error", MB_OK);
					break;
				}
				guest = false;
			}
			else if(GetWindowLong(hWnd, GWL_USERDATA) == 1) //Guest
			{
				strcpy(nameBuf, "GUEST");
				nameLen = strlen(nameBuf);
				strcpy(pass, "");
				passLen = strlen(pass);
				guest = true;
			}
			else
			{
				strcpy(nameBuf, "ADMIN");
				nameLen = strlen(nameBuf);

				GetDlgItemText(hWnd, IDC_PASSWORD, pass, 255);
				passLen = strlen(pass);
				if(passLen == 0)
				{
					MessageBox(hWnd, "Please enter a password.", "Error", MB_OK);
					break;
				}
				guest = false;
			}
			
			unsigned long msgLen;
			msgLen = 1;
			msgLen += sizeof(nameLen);
			msgLen += nameLen;
			msgLen += sizeof(passLen);
			msgLen += passLen;

			server->send(htonl(msgLen));
			server->send(ICV_OP_LOGIN);
			server->send(nameLen);
			server->send(nameBuf);
			server->send(passLen);
			server->send(pass);

			unsigned char result;
			server->receive((char*)&result, 1);
			if(result == ICV_SR_BADNAME)
			{
				MessageBox(hWnd, "Invalid username!", "Error", MB_OK);
				break;
			}
			else if(result == ICV_SR_BADPASS)
			{
				MessageBox(hWnd, "Password incorrect.", "Error", MB_OK);
				break;
			}
			else if(result != ICV_SR_OK)
			{
				MessageBox(hWnd, "Login error!", "Error", MB_OK);
				break;
			}

			if(GetWindowLong(hWnd, GWL_USERDATA) == 2)
			{   //If not guest or admin,
				//update Username history
				outFile.open("icvtemp.tmp");
				outFile << nameBuf << "\n";
				inFile.open("userhistory.dat");
				if(inFile.is_open())
				{
					char tempbuf[15];
					for(int i = 0; i < 5; ++i)
					{
						inFile.getline(tempbuf, 15);
						if(inFile.eof())
							break;
						outFile << tempbuf << "\n";
					}
					inFile.close();
				}
				outFile.close();
				DeleteFile("userhistory.dat");
				rename("icvtemp.tmp", "userhistory.dat");
			}

			convertToUpper(nameBuf);
			if(strcmp(nameBuf, "ADMIN") == 0)
				dlgState = DS_ADMIN;
			else
				dlgState = DS_FILES;

			EndDialog(hWnd, 0);
			break;
		case IDC_NEWUSER:
			if(DialogBox(hinst, MAKEINTRESOURCE(IDD_NEWUSER), hWnd, NewUserProc) == 0)
			{
				MessageBox(hWnd,"Now you can use your\nnew username and password\nto log onto the server.",
					"Note", MB_OK);
			}
			break;
		case IDC_DISCONNECT:
			server->send(htonl(1));
			server->send(ICV_OP_DISCONNECT);
			//backToConnect(hWnd);
			server->release();
			delete server;
			server = new Socket;
			dlgState = DS_CONNECT;
			EndDialog(hWnd, 0);
			break;
		case IDC_QUIT:
			quit = true;
			server->send(htonl(1));
			server->send(ICV_OP_DISCONNECT);
			EndDialog(hWnd,0);
			break;

		case IDC_GUEST:
			ctrl = GetDlgItem(hWnd, IDC_USERNAME);
			EnableWindow(ctrl, FALSE);
			ctrl = GetDlgItem(hWnd, IDC_PASSWORD);
			EnableWindow(ctrl, FALSE);
			SetWindowLong(hWnd, GWL_USERDATA, 1);
			break;
		case IDC_NORMAL:
			ctrl = GetDlgItem(hWnd, IDC_USERNAME);
			EnableWindow(ctrl, TRUE);
			ctrl = GetDlgItem(hWnd, IDC_PASSWORD);
			EnableWindow(ctrl, TRUE);
			SetWindowLong(hWnd, GWL_USERDATA, 2);
			break;
		case IDC_ADMIN:
			ctrl = GetDlgItem(hWnd, IDC_USERNAME);
			EnableWindow(ctrl, FALSE);
			ctrl = GetDlgItem(hWnd, IDC_PASSWORD);
			EnableWindow(ctrl, TRUE);
			SetWindowLong(hWnd, GWL_USERDATA, 3);
		}
		return true;
	}
	return false;
}
BOOL CALLBACK FilesProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	if(serverShutdown())
	{
		//MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		//backToConnect(hWnd);
		
		server->release();
		delete server;
		server = new Socket;
		MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
	}

	HWND ctrl;
	switch(msg)
	{
	case WM_INITDIALOG:
		SetDlgItemText(hWnd, IDC_STATICDIR, "c:\\");
		getDirList(hWnd, IDC_DIRLIST, IDC_STATICDIR, true);
		getDirList(hWnd, IDC_FILELIST, IDC_STATICDIR, false);

		if(!guest)
		{
			CheckRadioButton(hWnd, IDC_PERSONAL, IDC_SHARED, IDC_PERSONAL);
			folder = ICV_F_PERSONAL;
		}
		else
		{
			CheckRadioButton(hWnd, IDC_PERSONAL, IDC_SHARED, IDC_SHARED);
			folder = ICV_F_SHARED;

			ctrl = GetDlgItem(hWnd, IDC_PERSONAL);
			EnableWindow(ctrl, FALSE);
			setButtonStates(hWnd, FALSE);
		}
		ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
		checkingGetRmDirList(hWnd, ctrl, folder);
		
		return false;

	case WM_CLOSE:
		quit = true;
		server->send(htonl(1));
		server->send(ICV_OP_DISCONNECT);
		EndDialog(hWnd, 0);
		return true;
		
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_DIRLIST:
			if(HIWORD(wParam) == LBN_DBLCLK)
			{
				char path[MAX_PATH];
				GetDlgItemText(hWnd, IDC_STATICDIR, path, MAX_PATH);
				
				ctrl = (HWND)lParam;
				char buf[MAX_PATH];
				int index = (int)SendMessage(ctrl, LB_GETCURSEL, 0, 0);
				SendMessage(ctrl, LB_GETTEXT, index, (LPARAM)buf);

				std::string newPath = path;
				if(strcmp(buf, "..") == 0)
				{
					int pos = newPath.find_last_of("\\");
					newPath.erase(pos, newPath.size() - 1);
					if(newPath.compare("c:") == 0)
						newPath.append("\\");
				}
				else
				{
					if(newPath.compare("c:\\") != 0)
						newPath.append("\\");
					newPath.append(buf);
				}

				SetDlgItemText(hWnd, IDC_STATICDIR, newPath.c_str());
				getDirList(hWnd, IDC_DIRLIST, newPath.c_str(), true, true);
				getDirList(hWnd, IDC_FILELIST, newPath.c_str(), false, false);
			}
			break;

		case IDC_UPLOAD:
			uploadFile(hWnd);
			ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
			checkingGetRmDirList(hWnd, ctrl, folder);
			break;
		case IDC_DOWNLOAD:
			downloadFile(hWnd);
			getDirList(hWnd, IDC_FILELIST, IDC_STATICDIR, false);
			break;
		case IDC_RENAME:
			renameFile(hWnd);
			ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
			checkingGetRmDirList(hWnd, ctrl, folder);
			break;
		case IDC_DELETE:
			deleteFile(hWnd);
			ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
			checkingGetRmDirList(hWnd, ctrl, folder);
			break;
		case IDC_LOGOUT:
			server->send(htonl(1));
			server->send(ICV_OP_LOGOUT);
			dlgState = DS_LOGIN;
			folder = ICV_F_PERSONAL;
			EndDialog(hWnd, 0);
			break;
		case IDC_QUIT:
			quit = true;
			server->send(htonl(1));
			server->send(ICV_OP_DISCONNECT);
			EndDialog(hWnd, 0);
			break;

		case IDC_SETSHARE:
			dlgSetShare(hWnd);
			break;

		case IDC_SHARED:
			setButtonStates(hWnd, FALSE);
			folder = ICV_F_SHARED;
			ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
			checkingGetRmDirList(hWnd, ctrl, folder);
			break;
		case IDC_PERSONAL:
			setButtonStates(hWnd, TRUE);
			folder = ICV_F_PERSONAL;
			ctrl = GetDlgItem(hWnd, IDC_REMOTEFILES);
			checkingGetRmDirList(hWnd, ctrl, folder);
			break;
		}
		return true;
	}
	return false;
}
BOOL CALLBACK NewUserProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		EndDialog(hWnd, 1);
		return true;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			char nameBuf[256];
			unsigned char nameLen;
			char passBuf[256];
			unsigned char passLen;
			char passBuf2[256];
			unsigned char passLen2;

			GetDlgItemText(hWnd, IDC_USER, nameBuf, 256);
			nameLen = strlen(nameBuf);
			if(nameLen == 0)
			{
				MessageBox(hWnd, "Please enter a username.", "Error", MB_OK);
				break;
			}
			
			GetDlgItemText(hWnd, IDC_PASS1, passBuf, 256);
			passLen = strlen(passBuf);
			GetDlgItemText(hWnd, IDC_PASS2, passBuf2, 256);
			passLen2 = strlen(passBuf2);
			if(strcmp(passBuf, passBuf2) != 0)
			{
				MessageBox(hWnd, "Passwords do not match.", "Error", MB_OK);
				break;
			}
			unsigned long msgLen;
			msgLen = 1;
			msgLen += sizeof(nameLen);
			msgLen += nameLen;
			msgLen += sizeof(passLen);
			msgLen += passLen;

			server->send(htonl(msgLen));
			server->send(ICV_OP_NEWUSER);
			server->send(nameLen);
			server->send(nameBuf);
			server->send(passLen);
			server->send(passBuf);

			unsigned char res;
			server->receive((char*)&res, 1);
			if(res == ICV_SR_BADNAME)
			{
				MessageBox(hWnd, "Username already taken.", "Error", MB_OK);
				break;
			}
			EndDialog(hWnd, 0);

			break;
		case IDCANCEL:
			EndDialog(hWnd, 1);
			break;
		}
		return true;
	}
	return false;
}
BOOL CALLBACK TransferProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		SetWindowLong(hWnd, GWL_USERDATA, 3);
		server->send(ICV_S_CANCEL);
		return true;

	case WM_COMMAND:
		if(LOWORD(wParam) == IDC_CANCEL)
		{
			SetWindowLong(hWnd, GWL_USERDATA, 3);
			server->send(ICV_S_CANCEL);
		}
		return true;
	}
	return false;
}
BOOL CALLBACK RenameProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	HWND parent;
	HWND remoteListBox;
	int listboxIndex;
	char* fNameOld;
	unsigned char fNameLenOld;

	char* fNameNew;
	unsigned char fNameLenNew;

	switch(msg)
	{
	case WM_INITDIALOG:
		parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
		remoteListBox = GetDlgItem(parent, IDC_REMOTEFILES);
		listboxIndex = SendMessage(remoteListBox, LB_GETCURSEL, 0, 0);
		if(listboxIndex == LB_ERR)
		{
			EndDialog(hWnd, 2);
			return false;
		}
		fNameLenOld = SendMessage(remoteListBox, LB_GETTEXTLEN, listboxIndex, 0);

		fNameOld = new char[fNameLenOld + 1];
		SendMessage(remoteListBox, LB_GETTEXT, listboxIndex, (LPARAM)fNameOld);

		SetDlgItemText(hWnd, IDC_OLDNAME, fNameOld);
		delete[] fNameOld;
		return false;

	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return true;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
			remoteListBox = GetDlgItem(parent, IDC_REMOTEFILES);
			listboxIndex = SendMessage(remoteListBox, LB_GETCURSEL, 0, 0);
			fNameLenOld = SendMessage(remoteListBox, LB_GETTEXTLEN, listboxIndex, 0);
			
			fNameOld = new char[fNameLenOld + 1];
			SendMessage(remoteListBox, LB_GETTEXT, listboxIndex, (LPARAM)fNameOld);
			
			HWND edit;
			edit = GetDlgItem(hWnd, IDC_NEWNAME);
			fNameLenNew = SendMessage(edit, WM_GETTEXTLENGTH, 0, 0);
			fNameNew = new char[fNameLenNew + 1];
			SendMessage(edit, WM_GETTEXT, fNameLenNew + 1, (LPARAM)fNameNew);
			
			unsigned long msgLen;
			msgLen = 1;
			msgLen += sizeof(fNameLenOld);
			msgLen += fNameLenOld;
			msgLen += sizeof(fNameLenNew);
			msgLen += fNameLenNew;

			server->send(htonl(msgLen));

			server->send(ICV_OP_RENAME);
			server->send(fNameLenOld);
			server->send(fNameOld);
			server->send(fNameLenNew);
			server->send(fNameNew);
			delete[] fNameOld;
			delete[] fNameNew;

			unsigned char res;
			server->receive((char*)&res, 1);
			if(res == ICV_SR_OK)
				EndDialog(hWnd, 1);
			else
				EndDialog(hWnd, 2);
			break;
		case IDCANCEL:
			EndDialog(hWnd, 0);
			break;
		}
		return true;
	}
	return false;
}

BOOL CALLBACK ShareProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	char* fName;
	unsigned char len;
	unsigned long fileSize;
	unsigned char shared;
	unsigned long msgLen;
	HWND parent;

	switch(msg)
	{
	case WM_INITDIALOG:
		int index;
		parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
		index = SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETCURSEL, 0, 0);
		if(index == LB_ERR)
		{
			MessageBox(hWnd, "No remote file selected.", "Error", MB_OK);
			EndDialog(hWnd, 1);
			return false;
		}

		len = SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETTEXTLEN, index, 0);
		fName = new char[len + 1];

		SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETTEXT, index, (LPARAM)fName);

		msgLen = 1;
		msgLen += sizeof(len);
		msgLen += len;
		server->send(htonl(msgLen));

		server->send(ICV_OP_GETSHARE);
		server->send(len);
		server->send(fName);

		if(SERR(server->packetReceive((char*)&fileSize, 4)) ||
			SERR(server->receive((char*)&shared, 1)))
		{
			EndDialog(hWnd, 0);

			//MessageBox(parent, "Connection to server lost.", "Error", MB_OK);
			//backToConnect(parent);
			
			dlgState = DS_CONNECT;
			server->release();
			delete server;
			server = new Socket;

			EndDialog(parent, 0);
		}

		SetDlgItemText(hWnd, IDC_FILENAME, fName);
		delete[] fName;

		char chFileSize[19];

		itoa(ntohl(fileSize), chFileSize, 10);
		strcat(chFileSize, " bytes");
		SetDlgItemText(hWnd, IDC_FILESIZE, chFileSize);
		if(shared)
			CheckDlgButton(hWnd, IDC_SHARE, BST_CHECKED);

		SetWindowLong(hWnd, GWL_USERDATA, shared);

		return false;

	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return true;
		
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			UINT checkState;
			checkState = IsDlgButtonChecked(hWnd, IDC_SHARE);
			if((checkState == BST_CHECKED) != GetWindowLong(hWnd, GWL_USERDATA))
			{
				parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
				index = SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETCURSEL, 0, 0);
				len = SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETTEXTLEN, index, 0);
				
				fName = new char[len + 1];
				SendDlgItemMessage(parent, IDC_REMOTEFILES, LB_GETTEXT, index, (LPARAM)fName);
				
				sendShareState(fName, (unsigned char)(checkState == BST_CHECKED));
				delete[] fName;
			}
			EndDialog(hWnd, 0);
			break;
		case IDCANCEL:
			EndDialog(hWnd, 0);
			break;
		}
		return true;
	}

	return false;
}

BOOL CALLBACK AdminProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		quit = true;
		server->send(htonl(1));
		server->send(ICV_OP_DISCONNECT);
		EndDialog(hWnd, 0);
		return true;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_QUIT:
			quit = true;
			server->send(htonl(1));
			server->send(ICV_OP_DISCONNECT);
			EndDialog(hWnd, 0);
			break;
		case IDC_LOGOUT:
			server->send(htonl(1));
			server->send(ICV_OP_LOGOUT);
			dlgState = DS_LOGIN;
			folder = ICV_F_PERSONAL;
			EndDialog(hWnd, 0);
			break;

		case IDC_FILEFOLDER:
			dlgState = DS_ADMINFILES;
			EndDialog(hWnd, 0);
			break;

		case IDC_SHUTDOWN:
			server->send(htonl(1));
			server->send(ICV_OP_SHUTDOWN);
			//backToConnect(hWnd);
			
			server->release();
			delete server;
			server = new Socket;
			dlgState = DS_CONNECT;
			EndDialog(hWnd, 0);
			break;

		case IDC_GETUSERINFO:
			dlgState = DS_USERS;
			EndDialog(hWnd, 0);
			break;
		}
		return true;
	}
	return false;
}
BOOL CALLBACK AdminFilesProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		SetDlgItemText(hWnd, IDC_LOGOUT, "Done");
		return FilesProc(hWnd, msg, wParam, lParam);

	case WM_COMMAND:
		if(LOWORD(wParam) == IDC_LOGOUT)
		{
			dlgState = DS_ADMIN;
			EndDialog(hWnd, 0);
			return true;
		}
		else
			return FilesProc(hWnd, msg, wParam, lParam);

	default:
		return FilesProc(hWnd, msg, wParam, lParam);
	}

	return false;
}

BOOL CALLBACK UserInfoProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		getUserList(hWnd, IDC_USERLIST);
		return false;

	case WM_CLOSE:
		quit = true;
		server->send(htonl(1));
		server->send(ICV_OP_DISCONNECT);
		EndDialog(hWnd, 0);
		return true;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_DONE:
			dlgState = DS_ADMIN;
			EndDialog(hWnd, 0);
			break;
		case IDC_GETINFO:
			if(!getUserInfo(hWnd))
			{
				MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
				server->release();
				delete server;
				server = new Socket;
				dlgState = DS_CONNECT;
				HWND parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
				EndDialog(hWnd, 0);
				EndDialog(parent, 0);
			}
			break;
		case IDC_DROP:
			dropUser(hWnd, IDC_USERLIST);
			break;
		}
		return true;
	}
	return false;
}

//Function to check if server has closed
bool serverShutdown()
{
	if(server->hasData())
	{
		unsigned char temp;
		int bytes = server->receive((char*)&temp, 1);
		if(bytes == 0)
			return true;
		else if((bytes == SOCKET_ERROR) && (WSAGetLastError() != WSAEMSGSIZE))
			return true;
		else if(temp == ICV_OP_DISCONNECT)
			return true;
	}
	return false;
}

void setButtonStates(HWND hWnd, BOOL state)
{
	HWND button;
	button = GetDlgItem(hWnd, IDC_RENAME);
	EnableWindow(button, state);
	button = GetDlgItem(hWnd, IDC_DELETE);
	EnableWindow(button, state);
	button = GetDlgItem(hWnd, IDC_UPLOAD);
	EnableWindow(button, state);
	button = GetDlgItem(hWnd, IDC_SETSHARE);
	EnableWindow(button, state);
}

void sendShareState(const char* fName, unsigned char share)
{
	unsigned char len = strlen(fName);

	unsigned long msgLen = 1;
	msgLen += sizeof(len);
	msgLen += len;
	msgLen += sizeof(unsigned char);
	server->send(htonl(msgLen));
	
	server->send(ICV_OP_SETSHARE);
	server->send(len);
	server->send(fName);
	server->send(share);
}

void getDirList(HWND hWnd, int listboxID, int pathID, bool directory, bool prevs)
{
	char* temp;
	int len;

	HWND path = GetDlgItem(hWnd, pathID);
	len = SendMessage(path, WM_GETTEXTLENGTH, 0, 0);
	temp = new char[len + 1];
	SendMessage(path, WM_GETTEXT, len + 1, (LPARAM)temp);

	getDirList(hWnd, listboxID, temp, directory, prevs);

	delete[] temp;
}

void getDirList(HWND hWnd, int listboxID, const char* path, bool directory, bool prevs)
{
	WIN32_FIND_DATA wfd;

	std::string searchString;
	searchString = path;
	if(strcmp(path, "c:\\") != 0)
		searchString.append("\\");
	searchString.append("*.*");

	HWND listbox = GetDlgItem(hWnd, listboxID);
	SendMessage(listbox, LB_RESETCONTENT, 0, 0);

	HANDLE h = FindFirstFile(searchString.c_str(), &wfd);
	if(h == INVALID_HANDLE_VALUE)
		return;

	int res;
	do
	{
		if( (
				directory ^
				!(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			)
			&&
			(
				prevs || (strcmp(wfd.cFileName, "..") != 0)
			)
			&&
			strcmp(wfd.cFileName, ".") != 0
		   )
		{
			SendMessage(listbox, LB_ADDSTRING, 0, (LPARAM)wfd.cFileName);
		}

		res = FindNextFile(h, &wfd);
	}while(res != 0);

	FindClose(h);
}

void convertToUpper(char* str)
{
	int len = strlen(str);
	for(int i = 0; i < len; ++i)
	{
		str[i] = toupper(str[i]);
	}
}

//Upload file
void uploadFile(HWND hWnd)
{
	//Get the directory path
	char path[MAX_PATH];
	GetDlgItemText(hWnd, IDC_STATICDIR, path, MAX_PATH);

	//Get index of selected filename
	HWND lb = GetDlgItem(hWnd, IDC_LOCALFILES);
	int index = SendMessage(lb, LB_GETCURSEL, 0, 0);
	if(index == LB_ERR)
	{
		MessageBox(hWnd, "No file selected.", "Error", MB_OK);
		return;
	}

	//Get filename and add to path
	unsigned char fNameLen = SendMessage(lb, LB_GETTEXTLEN, index, 0);
	char* fName = new char[fNameLen + 1];
	SendMessage(lb, LB_GETTEXT, index, (LPARAM)fName);

	strcat(path, "\\");
	strcat(path, fName);
	
	//Get file size with Windows file i/o
	HANDLE h = CreateFile(path, GENERIC_READ, 0, NULL, OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, NULL);
	if(h == INVALID_HANDLE_VALUE)
	{
		MessageBox(hWnd, "Error opening file!", "Error", MB_OK);
		return;
	}
	unsigned long fileSize = GetFileSize(h, NULL);
	CloseHandle(h);

	//Open file with STL ifstream
	std::ifstream file(path, std::ios::in | std::ios::binary);
	if(!file)
		return;

	//Send the size of the message (header)
	unsigned long msgSize = 1;
	msgSize += sizeof(fNameLen);
	msgSize += fNameLen;
	msgSize += sizeof(unsigned long); //FILESEND_PACKETSIZE as unsigned long
	msgSize += sizeof(fileSize);

	server->send(htonl(msgSize));

	//Send the rest of the filereceive init data
	server->send(ICV_OP_TOSERVER);
	server->send(fNameLen);
	server->send(fName);
	server->send(htonl((unsigned long)FILESEND_PACKETSIZE));
	server->send(htonl(fileSize));

	delete[] fName;

	//Pop up the file-transfer dialog box and upload the file
	dlgUpload(hWnd, file, fileSize);

	file.close();
}

//Upload status dialog
void dlgUpload(HWND parent, std::ifstream& file, unsigned long fileSize)
{
	HWND dlg = CreateDialog(hinst, MAKEINTRESOURCE(IDD_TRANSFER),
		parent, TransferProc);
	ShowWindow(dlg, SW_SHOW);
	
	//Display the total bytes to be sent
	char* temp = new char[log(fileSize) + 8];
	ultoa(fileSize, temp, 10);
	strcat(temp, " bytes");
	SetDlgItemText(dlg, IDC_TOTALBYTES, temp);
	delete[] temp;

	int bytesSent;
	unsigned long totalBytesSent = 0;
	char chBytesSent[17];
	char packet[FILESEND_PACKETSIZE];
	HWND cProgress = GetDlgItem(dlg, IDC_PROGRESS);
	SendMessage(cProgress, PBM_SETRANGE32, 0, fileSize);

	//These variables used for displaying transfer rate
	int rate = 0;
	int last = GetTickCount();
	char chRate[20];

	//Loop: process messages and send file
	SetWindowLong(dlg, GWL_USERDATA, 1);

	Socket dataChannel;
	if(!dataChannel.connect(hostbuf, ICV_DATAPORT))
	{
		MessageBox(dlg, "Could not open data channel;\nDisconnecting from server.", "Error", MB_OK);
		DestroyWindow(dlg);
		//backToConnect(parent);

		server->release();
		delete server;
		server = new Socket;
		dlgState = DS_CONNECT;
		EndDialog(parent, 0);
		
		return;
	}

	MSG msg;
	while(GetWindowLong(dlg, GWL_USERDATA) == 1 && !file.eof())
	{
		if(serverShutdown())
		{
			server->release();
			delete server;
			server = new Socket;
			MessageBox(dlg, "Connection to server lost.", "Error", MB_OK);
			dlgState = DS_CONNECT;
			EndDialog(parent, 0);
			
			//MessageBox(dlg, "Connection to server lost.", "Error", MB_OK);
			//backToConnect(parent);

			SetWindowLong(dlg, GWL_USERDATA, 2);

			break;
		}

		//Process messages, ending loop if user has cancelled
		while(PeekMessage(&msg, dlg, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
		if(GetWindowLong(dlg, GWL_USERDATA) == 3)
		{
			break;
		}

		//Read the data into the buffer
		file.read(packet, FILESEND_PACKETSIZE);
		if(file.gcount() == 0)
			break;

		//Send the packet and increase the counters
		bytesSent = dataChannel.send(packet, file.gcount());
		totalBytesSent += bytesSent;
		rate += bytesSent;
		
		//Update the progress bar
		SendMessage(cProgress, PBM_DELTAPOS, (WPARAM)bytesSent, 0);

		//Update the "bytes sent" text
		ultoa(totalBytesSent, chBytesSent, 10);
		strcat(chBytesSent, " bytes");
		SetDlgItemText(dlg, IDC_TRANSFERRED, chBytesSent);

		//Update the "transfer rate" text every second
		if(GetTickCount() - last >= 1000)
		{
			ultoa((rate >> 10), chRate, 10);
			strcat(chRate, " kb/s");
			SetDlgItemText(dlg, IDC_RATE, chRate);
			rate = 0;
			last += 1000;
		}
	}

	dataChannel.release();

	unsigned char status;
	server->receive((char*)&status, 1);
	DestroyWindow(dlg);
}

//Download file
void downloadFile(HWND hWnd)
{
	//Get the directory path
	char path[MAX_PATH];
	GetDlgItemText(hWnd, IDC_STATICDIR, path, MAX_PATH);

	//Get index of selected filename
	HWND lb = GetDlgItem(hWnd, IDC_REMOTEFILES);
	int index = SendMessage(lb, LB_GETCURSEL, 0, 0);
	if(index == LB_ERR)
	{
		MessageBox(hWnd, "No file selected.", "Error", MB_OK);
		return;
	}

	//Get filename
	unsigned char fNameLen = SendMessage(lb, LB_GETTEXTLEN, index, 0);
	char* fName = new char[fNameLen + 1];
	SendMessage(lb, LB_GETTEXT, index, (LPARAM)fName);

	//Send the size of the message (header)
	unsigned long msgSize = 1;
	msgSize += sizeof(fNameLen);
	msgSize += fNameLen;
	msgSize += sizeof(unsigned long); //FILESEND_PACKETSIZE as unsigned long
	msgSize += sizeof(unsigned char); //Folder code as unsigned char

	server->send(htonl(msgSize));

	//Send the rest of the filereceive init data
	server->send(ICV_OP_TOCLIENT);
	server->send(fNameLen);
	server->send(fName);
	server->send(htonl(FILESEND_PACKETSIZE));
	server->send(folder);

	//Pop up the file-transfer dialog box and upload the file
	if(!dlgDownload(hWnd, path, fName))
	{
		
		server->release();
		delete server;
		server = new Socket;

		MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
		
		//MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		//backToConnect(hWnd);
	}
	delete[] fName;
}

//Download status dialog
bool dlgDownload(HWND parent, const char* path, std::string fName)
{
	unsigned long fileSize;
	if(SERR(server->packetReceive((char*)&fileSize, 4)))
		return false;
	
	fileSize = ntohl(fileSize);

	HWND dlg = CreateDialog(hinst, MAKEINTRESOURCE(IDD_TRANSFER),
		parent, TransferProc);
	ShowWindow(dlg, SW_SHOW);
	
	//Display the total bytes to be sent
	char* temp = new char[log(fileSize) + 8];
	ultoa(fileSize, temp, 10);
	strcat(temp, " bytes");
	SetDlgItemText(dlg, IDC_TOTALBYTES, temp);
	delete[] temp;

	//For dialog status updating
	char chBytesRcvd[17];
	HWND cProgress = GetDlgItem(dlg, IDC_PROGRESS);
	SendMessage(cProgress, PBM_SETRANGE32, 0, fileSize);

	//For rate calculating/updating
	int rate = 0;
	int last = GetTickCount();
	char chRate[20];

	//Receive-counters, data
	unsigned long fileBytesReceived = 0;
	char packet[FILESEND_PACKETSIZE];
	
	//Create the file, change \ to ~
	for(int i = 0; i < fName.size(); ++i)
	{
		if(fName[i] == '\\')
		{
			fName[i] = '~';
			break;
		}
	}
	std::string filePath;
	filePath = path;
	filePath.append("\\");
	filePath.append(fName);

	bool ok;
	std::ofstream file(filePath.c_str(), std::ios::out | std::ios::binary);
	if(!file)
		ok = false;
	else
		ok = true;
	
	Socket dataChannel;
	if(!dataChannel.connect(hostbuf, ICV_DATAPORT))
	{
		MessageBox(dlg, "Could not open data channel;\nDisconnecting from server.", "Error", MB_OK);
		DestroyWindow(dlg);
		//backToConnect(parent);
		
		server->release();
		delete server;
		server = new Socket;
		dlgState = DS_CONNECT;
		DestroyWindow(dlg);
		EndDialog(parent, 0);
				
		return true;
	}

	MSG msg;

	//The receive/update loop
	while(ok)
	{
		//Handle the messages first
		while(PeekMessage(&msg, dlg, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}

		//If the server has sent more data, process it
		if(dataChannel.hasData())
		{
			int bytes = dataChannel.receive(packet,
				FILESEND_PACKETSIZE);
			
			if(bytes == SOCKET_ERROR)
			{
				file.close();
				DestroyWindow(dlg);
				dataChannel.release();
				return false;
			}
			else if(bytes == 0)
				break;
			
			file.write(packet, bytes);

			fileBytesReceived += bytes;
			if(fileBytesReceived == fileSize)
				break;

			rate += bytes;

			//Update the progress bar
			SendMessage(cProgress, PBM_DELTAPOS, (WPARAM)bytes, 0);

			//Update the "bytes transferred" text
			ultoa(fileBytesReceived, chBytesRcvd, 10);
			strcat(chBytesRcvd, " bytes");
			SetDlgItemText(dlg, IDC_TRANSFERRED, chBytesRcvd);
		}

		//Update the "transfer rate" text every second
		if(GetTickCount() - last >= 1000)
		{
			ultoa((rate >> 10), chRate, 10);
			strcat(chRate, " kb/s");
			SetDlgItemText(dlg, IDC_RATE, chRate);
			rate = 0;
			last += 1000;
		}
	}

	file.close();
	dataChannel.release();
	DestroyWindow(dlg);
	return true;
}

void deleteFile(HWND hWnd)
{
	HWND lb = GetDlgItem(hWnd, IDC_REMOTEFILES);

	int index = SendMessage(lb, LB_GETCURSEL, 0, 0);
	if(index == LB_ERR)
	{
		MessageBox(hWnd, "No file selected.", "Error", MB_OK);
		return;
	}

	unsigned char nameLen = SendMessage(lb, LB_GETTEXTLEN, index, 0);
	char* fNameBuf = new char[nameLen + 1];
	SendMessage(lb, LB_GETTEXT, index, (LPARAM)fNameBuf);

	sendShareState(fNameBuf, 0);

	unsigned long msgLen = 1;
	msgLen += sizeof(nameLen);
	msgLen += nameLen;
	server->send(htonl(msgLen));

	server->send(ICV_OP_DELETE);
	server->send(nameLen);
	server->send(fNameBuf);

	unsigned char result;
	server->receive((char*)&result, 1);

	if(result == ICV_SR_ERR)
		MessageBox(hWnd, "File could not be deleted.", "Error", MB_OK);
	else if(result == ICV_SR_OK)
		MessageBox(hWnd, "File successfully deleted.", "Success", MB_OK);
	else
		MessageBox(hWnd, "Unknown protocol error.", "Error", MB_OK);

	delete[] fNameBuf;
}

void renameFile(HWND hWnd)
{
	int res = DialogBox(hinst, MAKEINTRESOURCE(IDD_RENAME), hWnd, RenameProc);
	if(res == 1)
		MessageBox(hWnd, "File successfully renamed.", "Success", MB_OK);
	else if(res == 2)
		MessageBox(hWnd, "File could not be renamed.", "Error", MB_OK);
}

//Function to update the "Remote Files" listbox
bool getRemoteDirList(HWND lbControl, unsigned char folder)
{
	SendMessage(lbControl, LB_RESETCONTENT, 0, 0);

	server->send(htonl(2));
	server->send(ICV_OP_GETDIRLIST);
	server->send(folder);

	int bytes;
	int bytesReceived;
	char name[256];
	unsigned char temp;

	while(true)
	{
		server->receive((char*)&temp, 1); //Receive status
		if(SERR(temp))
			return false;

		if(temp != ICV_S_CONTINUE) //If status is "end"
			break;

		server->receive((char*)&temp, 1); //Receive length
		if(SERR(temp))
			return false;

		bytesReceived = 0;
		do
		{
			bytes = server->receive(&name[bytesReceived],
				temp - bytesReceived);

			if(SERR(bytes))
				return false;

			bytesReceived += bytes;
		}while(bytesReceived < temp);
		name[temp] = '\0';
		
		SendMessage(lbControl, LB_ADDSTRING, 0, (LPARAM)name);
	}
	return true;
}

void getUserList(HWND hWnd, int listboxID)
{
	HWND ctrl = GetDlgItem(hWnd, listboxID);

	server->send(htonl(1));
	server->send(ICV_OP_GETUSERLIST);

	unsigned char nameLen, status;
	char name[256];

	bool ok = true;

	if(SERR(server->receive((char*)&status, 1)))
		ok = false;

	while(ok && status != ICV_S_END)
	{
		if(SERR(server->receive((char*)&nameLen, 1)))
		{
			ok = false;
			break;
		}
		if(SERR(server->packetReceive(name, nameLen)))
		{
			ok = false;
			break;
		}

		name[nameLen] = '\0';
		SendDlgItemMessage(hWnd, listboxID, LB_ADDSTRING, 0, (LPARAM)name);

		if(SERR(server->receive((char*)&status, 1)))
		{
			ok = false;
			break;
		}
	}
	if(!ok)
	{
		HWND parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
		server->release();
		delete server;
		server = new Socket;
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
		MessageBox(parent, "Connection to server lost.", "Error", MB_OK);
		EndDialog(parent, 0);
	}
}

bool getUserInfo(HWND hWnd)
{
	int index = SendDlgItemMessage(hWnd, IDC_USERLIST, LB_GETCURSEL, 0, 0);
	if(index == LB_ERR)
	{
		MessageBox(hWnd, "No user selected.", "Error", MB_OK);
		return true;
	}

	unsigned char len = SendDlgItemMessage(hWnd, IDC_USERLIST, LB_GETTEXTLEN, index, 0);
	char* name = new char[len + 1];
	SendDlgItemMessage(hWnd, IDC_USERLIST, LB_GETTEXT, index, (LPARAM)name);

	unsigned long msgLen = 1;
	msgLen += sizeof(len);
	msgLen += len;
	server->send(htonl(msgLen));

	server->send(ICV_OP_GETUSERINFO);
	server->send(len);
	server->send(name);

	unsigned char res;
	if(SERR(server->receive((char*)&res, 1)))
		return false;

	if(res == ICV_SR_BADNAME)
	{
		MessageBox(hWnd, "User has already disconnected.", "Error", MB_OK);
		SendDlgItemMessage(hWnd, IDC_USERLIST, LB_DELETESTRING, index, 0);
		return true;
	}
	
	unsigned char ipLen;
	if(SERR(server->receive((char*)&ipLen, 1)))
		return false;

	char* ipStr = new char[ipLen + 1];
	if(SERR(server->packetReceive(ipStr, ipLen)))
	{
		delete[] ipStr;
		return false;
	}
	ipStr[ipLen] = '\0';

	unsigned long filesShared;
	if(SERR(server->packetReceive((char*)&filesShared, 4)))
	{
		delete[] ipStr;
		return false;
	}
	filesShared = ntohl(filesShared);

	DWORD connectedTime;
	if(SERR(server->packetReceive((char*)&connectedTime, 4)))
	{
		delete[] ipStr;
		return false;
	}
	connectedTime = ntohl(connectedTime);

	unsigned char transferType;
	if(SERR(server->receive((char*)&transferType, 1)))
	{
		delete[] ipStr;
		return false;
	}

	unsigned long transferRate;
	unsigned long fileSize;
	unsigned long bytesTransferred;

	if(transferType != 1) //1 for CSTT_NONE
	{
		if(SERR(server->packetReceive((char*)&transferRate, 4)))
		{
			delete[] ipStr;
			return false;
		}
		if(SERR(server->packetReceive((char*)&fileSize, 4)))
		{
			delete[] ipStr;
			return false;
		}
		if(SERR(server->packetReceive((char*)&bytesTransferred, 4)))
		{
			delete[] ipStr;
			return false;
		}
		transferRate = ntohl(transferRate);
		fileSize = ntohl(fileSize);
		bytesTransferred = ntohl(bytesTransferred);
	}

	SetDlgItemText(hWnd, IDC_STATICIP, ipStr);
	char temp[64];
	itoa(filesShared, temp, 10);
	SetDlgItemText(hWnd, IDC_STATICSHARED, temp);
	itoa(connectedTime / 1000, temp, 10);
	strcat(temp, " seconds");
	SetDlgItemText(hWnd, IDC_STATICTIME, temp);
	switch(transferType)
	{
	case 1:
		SetDlgItemText(hWnd, IDC_STATICTTYPE, "None");
		break;
	case 2:
		SetDlgItemText(hWnd, IDC_STATICTTYPE, "To Server");
		break;
	case 3:
		SetDlgItemText(hWnd, IDC_STATICTTYPE, "To Client");
		break;
	}

	if(transferType != 1) //1 for CSTT_NONE
	{
		itoa(transferRate, temp, 10);
		strcat(temp, " kb/s");
		SetDlgItemText(hWnd, IDC_STATICRATE, temp);
		itoa(bytesTransferred, temp, 10);
		strcat(temp, " bytes");
		SetDlgItemText(hWnd, IDC_STATICSIZE, temp);
		itoa(bytesTransferred, temp, 10);
		strcat(temp, " bytes");
		SetDlgItemText(hWnd, IDC_STATICTRANSD, temp);
	}
	else
	{
		SetDlgItemText(hWnd, IDC_STATICRATE, "");
		SetDlgItemText(hWnd, IDC_STATICSIZE, "");
		SetDlgItemText(hWnd, IDC_STATICTRANSD, "");
	}

	delete[] name;
	delete[] ipStr;
	return true;
}

void dropUser(HWND hWnd, int listboxID)
{
	int index = SendDlgItemMessage(hWnd, listboxID, LB_GETCURSEL, 0,0);
	if(index == LB_ERR)
	{
		MessageBox(hWnd, "No user selected.", "Error", MB_OK);
		return;
	}
	unsigned char nameLen = SendDlgItemMessage(hWnd, listboxID, LB_GETTEXTLEN, index, 0);
	char* name = new char[nameLen + 1];

	SendDlgItemMessage(hWnd, listboxID, LB_GETTEXT, index, (LPARAM)name);

	unsigned long msgLen = 1;
	msgLen += sizeof(nameLen);
	msgLen += nameLen;
	server->send(htonl(msgLen));
	
	server->send(ICV_OP_DROPUSER);
	server->send(nameLen);
	server->send(name);

	unsigned char status;
	if(SERR(server->receive((char*)&status, 1)))
	{
		MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		HWND parent = (HWND)GetWindowLong(hWnd, GWL_HWNDPARENT);
		server->release();
		delete server;
		server = new Socket;
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
		EndDialog(parent, 0);
		return;
	}

	//SendDlgItemMessage(hWnd, listboxID, LB_DELETESTRING, index, 0);
	if(status == ICV_SR_ERR)
	{
		MessageBox(hWnd, "User already disconnected.", "Error", MB_OK);
		SendDlgItemMessage(hWnd, listboxID, LB_DELETESTRING, index, 0);
	}
	else if(status == ICV_SR_ERR2)
		MessageBox(hWnd, "Cannot drop self.", "Error", MB_OK);
	else //ICV_SR_OK
		SendDlgItemMessage(hWnd, listboxID, LB_DELETESTRING, index, 0);

	delete[] name;
}

void checkingGetRmDirList(HWND hWnd, HWND lbControl, unsigned char folder)
{
	if(!getRemoteDirList(lbControl, folder))
	{
		MessageBox(hWnd, "Connection to server lost.", "Error", MB_OK);
		//backToConnect(hWnd);
		server->release();
		delete server;
		server = new Socket;
		dlgState = DS_CONNECT;
		EndDialog(hWnd, 0);
	}
}
/*
void backToConnect(HWND hWndToClose)
{
	server->release();
	delete server;
	server = new Socket;
	dlgState = DS_CONNECT;
	EndDialog(hWndToClose, 0);
}
*/
//+-------------------------------+
//|   *Init/Shutdown Functions*   |
//+-------------------------------+

void appInit()
{
	Socket::networkInit();
	server = new Socket;
}

void appShutdown()
{
	server->release();
	delete server;
	Socket::networkShutdown();
}