/*
  cf.c - Column Format: format a file into multiple columns.

  Jason Hood, 11 to 14 August, 2006.

  An alternative to WORDY's RF utility, this utility will place the words into
  columns, instead of rows.
*/

#define PVERS "1.00"
#define PDATE "14 August, 2006"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined( _WIN32 )
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <conio.h>
#include <io.h>
#elif defined( __DJGPP__ )
#include <pc.h>
#include <unistd.h>
#undef __UNIX__
#elif defined( __MSDOS__ )
#include <bios.h>
#include <io.h>
#else
#include <unistd.h>
#ifndef __UNIX__
#define __UNIX__
#endif
#endif

//#include <stdbool.h>
#define bool  char
#define true  1
#define false 0


#define MAX_WORD 100	// UKACD 1.7 has an 89-character phrase

struct option
{
  int	width;			// number of characters across
  int	height; 		// number of lines for each column
  bool	pause;			// pause each screen full?
  bool	blank;			// add a blank line after each page
  char* name;			// name of file to format
}
option =
{
  -1,				// auto-detect width
  -1,				// auto-detect height
  true, 			// pause
  true, 			// add a blank line after each page
  NULL, 			// read standard input
};


struct wlist			// linked list of words
{
  struct wlist* next;
  char	 word[1];		// dynamically allocated
};
struct wlist* head;
int    max;			// longest word read

#ifdef __UNIX__
FILE* tty;
#endif


void help( bool ver )
#ifdef __GNUC__
__attribute__((noreturn))
#endif
;

void process_options( int argc, char* argv[] );
void get_screen_dims( int* width, int* height );
long create( FILE* file );
void display( long cnt );


int main( int argc, char* argv[] )
{
  FILE* file;
  long	cnt;

  process_options( argc, argv );
  if (option.width == -1 || option.height == -1)
  {
    int width, height;
    get_screen_dims( &width, &height );
    if (option.width == -1)
    {
      option.width = width;
    }
    if (option.height == -1)
    {
      option.height = height - option.blank;
    }
  }
  if (!isatty( 1 ))
  {
    option.pause = false;
  }

  if (option.name)
  {
    file = fopen( option.name, "r" );
    if (file == NULL)
    {
      printf( "Unable to open \"%s\".\n", option.name );
      return 1;
    }
  }
  else
  {
    file = stdin;
  }

#ifdef __UNIX__
  if (isatty( 0 ))
  {
    tty = stdin;
  }
  else
  {
    tty = fopen( "/dev/tty", "r" );
    if (tty == NULL)
    {
      tty = stderr;
    }
  }
#endif

  cnt = create( file );
  display( cnt );

  return 0;
}


long create( FILE* file )
{
  char	 word[MAX_WORD];
  struct wlist* w;
  struct wlist* tail;
  long	 cnt;

  head = tail = NULL;
  cnt  = max  = 0;
  while (fgets( word, sizeof(word), file ))
  {
    int len = strlen( word );
    if (word[len-1] != '\n')
    {
      static bool trunc = false;
      if (!trunc)
      {
	fputs( "Warning: word truncated.\n", stderr );
	trunc = true;
      }
    }
    else
    {
      word[--len] = '\0';
    }
    w = malloc( sizeof(struct wlist) + len );
    if (w == NULL)
    {
      display( cnt );
      w = malloc( sizeof(struct wlist) + len );
      if (w == NULL)
      {
	fputs( "Error: out of memory.\n", stderr );
	exit( 1 );
      }
      cnt = 0;
    }
    w->next = NULL;
    memcpy( w->word, word, len+1 );
    if (head == NULL)
    {
      head = w;
    }
    else
    {
      tail->next = w;
    }
    tail = w;
    ++cnt;
    if (len > max)
    {
      max = len;
    }
  }

  return cnt;
}


void display( long cnt )
{
  int	 cols, len;
  long	 line, lines, row[30];
  int	 wid[30], width;
  struct wlist* wrd[30];
  struct wlist* w;
  struct wlist* hd;
  long	 n, t;
  int	 c, j;
  bool	 done;

  hd = head;
  t  = cnt;
  while (cnt)
  {
    cols = (option.width + 3) / (max + 3);
    if (cols > 30)
    {
      cols = 30;
    }
    done = false;
    while (true)
    {
      lines = cnt / cols;
      j = (int)(cnt % cols);
      if (option.height > 1 && lines + ((j) ? 1 : 0) > option.height)
      {
	lines = option.height;
	j = 0;
      }
      for (c = cols; --c >= 0;)
      {
	row[c] = lines;
      }
      for (c = 0; c < j; ++c)
      {
	++row[c];
      }
      if (j)
      {
	++lines;
      }

      memset( wid, 0, sizeof(wid) );
      memset( wrd, 0, sizeof(wrd) );
      n = line = c = 0;
      width = -3;
      for (w = hd; w; w = w->next)
      {
	if (line == 0)
	{
	  wrd[c] = w;
	}
	len = strlen( w->word );
	if (len > wid[c])
	{
	  wid[c] = len;
	}
	++n;
	if (++line == row[c])
	{
	  line = 0;
	  width += wid[c] + 3;
	  if (++c == cols)
	  {
	    break;
	  }
	}
      }
      if (width < option.width)
      {
	if (done || cols == 30)
	{
	  break;
	}
	++cols;
      }
      else
      {
	--cols;
	done = true;
      }
    }

    cnt -= n;

    do
    {
      for (c = 0; c < cols; ++c)
      {
	printf( "%-*s", wid[c], wrd[c]->word );
	if (c != cols-1)
	{
	  fputs( "   ", stdout );
	}
	wrd[c] = wrd[c]->next;
	if (--n == 0)
	{
	  break;
	}
      }
      putchar( '\n' );
    } while (n > 0);

    hd = wrd[cols-1];

    if (cnt || (option.pause && option.height == lines))
    {
      if (option.pause)
      {
	char ch;
#ifdef __UNIX__
	printf( "[%ld/%ld] Press Enter to continue...", t - cnt, t );
	fflush( stdout );
	ch = getc( tty );
	fputs( "\x1b[A\x1b[K", stdout );
#else
	printf( "[%ld/%ld] Press any key to continue...", t - cnt, t );
# if defined( __WIN32__ )
	ch = _getch();
# elif defined( __DJGPP__ )
	fflush( stdout );
	ch = getkey();
# else
	ch = (char)_bios_keybrd( 0 );
# endif
	fputs( "\r                                            ", stdout );
#endif
	if (ch == 27 || ch == 'q')
	{
	  exit( 0 );
	}
	if (cnt)
	{
	  putchar( '\n' );
	}
      }
      else if (option.blank)
      {
	putchar( '\n' );
      }
    }
  }

  for (w = head; w;)
  {
    struct wlist* t = w->next;
    free( w );
    w = t;
  }
  head = NULL;
}


void get_screen_dims( int* width, int* height )
{
#if defined( _WIN32 )
  CONSOLE_SCREEN_BUFFER_INFO csbi;

  if (!GetConsoleScreenBufferInfo( GetStdHandle( STD_ERROR_HANDLE ), &csbi ))
  {
    *width  = 80;
    *height = 25;
  }
  else
  {
    *width  = csbi.dwSize.X;
    *height = csbi.srWindow.Bottom - csbi.srWindow.Top + 1;
  }

#elif defined( __DJGPP__ )
  *width  = ScreenCols();
  *height = ScreenRows();

#elif defined( __MSDOS__ )
  asm push ds
  asm push es
  asm push bx
  asm xor  ax, ax
  asm mov  es, ax
  asm mov  ax, es:[0x44a]
  asm lds  bx, width
  asm mov  [bx], ax
  asm mov  al, es:[0x484]
  asm inc  ax
  asm lds  bx, height
  asm mov  [bx], ax
  asm pop  bx
  asm pop  es
  asm pop  ds

#else
  char* env;

  env = getenv( "COLUMNS" );
  *width  = (env == NULL) ? 80 : atoi( env );
  env = getenv( "LINES" );
  *height = (env == NULL) ? 25 : atoi( env );
#endif
}


void process_options( int argc, char* argv[] )
{
  int arg;

  if (argc == 1)
  {
    return;
  }

  if (strcmp( argv[1], "--help" ) == 0)
  {
    help( false );
  }
  if (strcmp( argv[1], "--version" ) == 0)
  {
    help( true );
  }

  for (arg = 1; arg < argc; ++arg)
  {
    if (argv[arg][0] == '-' || argv[arg][0] == '/')
    {
      int j;
      for (j = 1; argv[arg][j]; ++j)
      {
	switch (argv[arg][j])
	{
	  case '?': help( false );

	  case 'c': option.pause = false; break;
	  case 'f': option.blank = false; break;

	  case 'w':
	  case 'h':
	  {
	    char* num;
	    int* val = (argv[arg][j] == 'w') ? &option.width : &option.height;
	    if (argv[arg][j+1] == '\0')
	    {
	      if (argv[arg+1] == NULL)
	      {
		puts( "Error: missing value." );
		exit( 1 );
	      }
	      num = argv[++arg];
	    }
	    else
	    {
	      num = argv[arg]+j+1;
	    }
	    *val = atoi( num );
	  }
	  goto next;

	  case '/': break;

	  default:
	    printf( "Error: unknown option '%c'.\n", argv[arg][j] );
	    exit( 1 );
	  break;
	}
      }
    }
    else
    {
      option.name = argv[arg];
    }
  next: ;
  }
}


void help( bool ver )
{
  puts( "Column Format by Jason Hood <jadoxa@yahoo.com.au>.\n"
	"Version "PVERS" ("PDATE").  Freeware.\n"
	"http://fw.adoxa.cjb.net/"
      );
  if (!ver)
  puts( "\n"
	"Format a list into a set of columns.\n"
	"\n"
	"cf [-cf] [-w WIDTH] [-h HEIGHT] [file]\n"
	"\n"
	"-c        continuous display - don't pause at end of page\n"
	"-f        flow - don't add a blank line between pages\n"
	"-w        use WIDTH characters across\n"
	"-h        use HEIGHT lines for each column\n"
	"file      the file to format (default is standard input)\n"
	"\n"
	"The default width and height is the screen width and height;\n"
	"use -h0 (or -h1) for a single page."
      );

  exit( 0 );
}
