package ccmenu;

import java.io.*;
import java.util.*;
import java.util.zip.CRC32;

/**
 * This class holds the database of roms and their CRC numbers for lookup
 *
 *
 */
public class CC2Database
{
    /** HashMap of all rom entries, stored by CRC key */
    private HashMap myRomDBHash = new HashMap();

    /** The max length of a game name/description  */
   static final int MAX_GAME_NAME_LEN = 40;

   /** The max length of a FILE name */
   final int MAX_FILE_NAME_LEN = 8;

    /**
     * Constructer: reads in the database file
     *
     * The input text file CC2ROMDB.TXT will have each rom entry on one line
     * and the values separated by one or more tab characters.
     * Example line from the CC2ROMDB.TXT file:
     * "3D Tic-Tac-Toe\t\t\t\t\t3DTIC\t\t2K\t\t78BIOS\t\tb98f339e"
     * @throws Exception
     */
    public CC2Database()
            throws Exception
    {
        BufferedReader inBuffer=null;
        File dbfile = new File("CC2ROMDB.TXT");
        try {
           inBuffer=new BufferedReader(new FileReader(dbfile));
        }
        catch (FileNotFoundException ex)
        {
            throw new Exception("File " + dbfile.getAbsolutePath() +
                                " not found.");
        }

        //read each line, storing each line as one entry in the hash
        String aLine;
        String[] entries;
        int lineNum = 0;
        RomEntry aRomEntry;
        try
        {
           while ( (aLine=inBuffer.readLine()) != null )
           {
              aRomEntry = new RomEntry(aLine);
              lineNum++;
              if(null != aRomEntry)
              {
                 myRomDBHash.put(aRomEntry.getCRC(), aRomEntry);
              }
              else
              {
                 throw new Exception("File " + dbfile.getAbsolutePath() +
                                     " has a bad line:" + aLine + ":");
              }
           }
        }
        catch (IOException ex)
        {
           throw new Exception("Can not read file " + dbfile.getAbsolutePath());
        }
        finally
        {
            try
            {
               inBuffer.close();
            }
            catch(IOException ex)
            {
            }
        }
    }


    /**
     * Takes a CRCString and looks in this DB for entry if any
     * @param CRCString the crc to lookup
     * @return a RomEntry of this CRCString, null if none exists.
     */
    public RomEntry lookup(String CRCString)
    {
        return (RomEntry) myRomDBHash.get(CRCString);
    }


    /**
     * Returns true if DB has this CRC key
     * @param CRCString the CRC key to check
     * @return true if DB has this CRC key
     */
    public boolean containsKey(String CRCString)
    {
       return myRomDBHash.containsKey(CRCString);
    }


    /**
     * Adds the new entry to the database
     * @param newEntry RomEntry
     */
    public void addEntry(RomEntry newEntry)
    {
      if(null != newEntry)
      {
        myRomDBHash.put(newEntry.getCRC(), newEntry);
      }
    }


    /**
     * Find all included entries in the database and return a sorted list of
     * them.
     * @return an sorted Iterator of all included entries.
     */
    public Iterator getAllIncluded()
    {
        ArrayList rtnList = new ArrayList();
        Iterator romIterator = myRomDBHash.values().iterator();
        RomEntry currEntry = null;
        while(romIterator.hasNext())
        {
            currEntry = (RomEntry) romIterator.next();
            if(currEntry.isIncluded() && currEntry.isListed())
            {
                rtnList.add(currEntry);
            }
        }
        // sort
        java.util.Collections.sort(rtnList);
        return rtnList.iterator();
    }


    /**
     * Find all NOT included entries in the database and return a sorted list of
     * them.
     * @return an sorted Iterator of all NOT included entries.
     */
    public Iterator getAllNotIncluded()
    {
        ArrayList rtnList = new ArrayList();
        Iterator romIterator = myRomDBHash.values().iterator();
        RomEntry currEntry = null;
        while(romIterator.hasNext())
        {
            currEntry = (RomEntry) romIterator.next();
            if(!currEntry.isIncluded())
            {
                rtnList.add(currEntry);
            }
        }
        // sort
        java.util.Collections.sort(rtnList);
        return rtnList.iterator();
    }


    /**
     * @return an Iterator in alphabetaical order.
     */
    public Iterator getAlphIterator()
    {
       ArrayList rtnList = new ArrayList();
       Iterator romIterator = myRomDBHash.values().iterator();
       RomEntry currEntry = null;
       while(romIterator.hasNext())
       {
           currEntry = (RomEntry) romIterator.next();
           if(!currEntry.isIncluded())
           {
               rtnList.add(currEntry);
           }
       }
       // sort
       java.util.Collections.sort(rtnList);

       return rtnList.iterator();
    }


    /**
     * Writes the complete database to the outFile
     * @param outFile place to write the output
     * @throws Exception
     */
    public void writeToFile(File outFile)
    throws Exception
    {
       Iterator romIterator = getAlphIterator();
       RomEntry currEntry = null;
      BufferedWriter txtOutput = null;
      try
      {
          txtOutput =
               new BufferedWriter(new FileWriter(outFile));

           while(romIterator.hasNext())
           {
             currEntry = (RomEntry) romIterator.next();
             txtOutput.write(currEntry.getDBEntry()+ "\n");
           }
           txtOutput.close();
      }
      catch (IOException ex)
      {
         throw new Exception("Can't write new DB file to " +
                             outFile.getAbsolutePath());
      }
    }
}
