package ccmenu;

import java.util.*;
import java.io.*;


/**
 * This Class converts a special input CC2 Menu text file into a
 * ouput CC2 Menu file used by the CC2's software.
 *
 * The input text menu file will have each value separated by one or more
 * tab characters and   values defined only as "-" (a dash character) will be
 * treated as blank.
 * Example line from the file: "3D Tic-Tac-Toe\t\t\t3DTIC\t\t2K\t\t78BIOS\t"
 * The output CC2 menu file is defined by field length, and will not have
 * any carriage returns or line feeds.
 * Example one game and the start of  another:
 * "3D Tic-Tac-Toe                          3DTIC   2K      78BIOS  A-Te"
 */
public class CC2MenuConvert
{
    /** used to count the number of error's in lines */
    private int myCurrentLineError=0;

    /** The max length of a game name/discription */
   static final int MAX_GAME_NAME_LEN = 40;

   /** The max length of a FILE name */
   final int MAX_FILE_NAME_LEN = 8;


   /**
    * Constructer
    */
   public CC2MenuConvert() {
   }


   int convert_menu(String InputFilename, String OutputFilename,
                    PrintStream errOut, PrintStream warnOut)
   {
      String gamename;
      String romname;
      String bankswitch;
      String startname;
      String aLine;
      int result = 1;
      int lineNum = 0;
      String[] entrys;

      BufferedReader in=null;
      BufferedWriter outfile=null;
      File inputFile = new File(InputFilename);
      StringBuffer outfileName = new StringBuffer(inputFile.getName());
      outfileName.delete(outfileName.indexOf("."),outfileName.length());
      outfileName.append(".CC2");
      OutputFilename = OutputFilename+"/"+outfileName;

      try {
         in=new BufferedReader(new FileReader(InputFilename));
      }
      catch (FileNotFoundException ex)
      {
            errOut.println("File " + InputFilename +" not found.");
            return result;
      }
      try {
         outfile=new BufferedWriter(new FileWriter(OutputFilename));
      }
      catch (IOException ex)
      {
            errOut.println("Unable to created file " + OutputFilename +".");
            return result;
      }
      try
      {
         while ( (aLine=in.readLine()) != null )
         {
            lineNum++;
            myCurrentLineError = 0;
            entrys = aLine.split("\\t+");
            gamename   = check_length(entrys[0],MAX_GAME_NAME_LEN);
            romname    = check_length(entrys[1],MAX_FILE_NAME_LEN);
            bankswitch = check_length(entrys[2],MAX_FILE_NAME_LEN);
            startname  = check_length(entrys[3],MAX_FILE_NAME_LEN);

            romname = check_filename(romname,"rom name",lineNum,errOut,warnOut);
            bankswitch = check_filename(bankswitch,"bankswitch file name",
                                        lineNum,errOut,warnOut);
            startname = check_filename(startname,"startup file name",
                                       lineNum,errOut,warnOut);

            if(myCurrentLineError > 0)
            {
                errOut.println("Line number: "+lineNum +
                       ": "+aLine);
                continue; // discard this line and keep going
            }

            // write one game to the outfile
            outfile.write(gamename);
            outfile.write(romname);
            outfile.write(bankswitch);
            outfile.write(startname);
         }
         result = 0;
      }
      catch (IOException ex)
      {
          errOut.println("Error: an IOException");
      }
      finally
      {
          try
          {
             in.close();
             outfile.close();
          }
          catch(IOException ex)
          {
             errOut.println("Error: an IOException");
          }
      }

      return result;
   }


   /**
    * Check a filename empty or dash, has only valid filename charaters.
    * @param filename the String to check
    * @param fileDisc the discription of this file name to use for err/warn
    * @param lineNum the current line number
    * @param errOut the error output PrintStream
    * @param warnOut the warning output PrintStream
    * @return the newline formed String
    */
   private String check_filename(String filename, String fileDisc, int lineNum,
                                 PrintStream errOut, PrintStream warnOut)
   {
       if(filename.trim().equals("-") ||
          filename.trim().length() < 1)
       {
           warnOut.println("Warning: Line number:"+lineNum+
                           " has a empty "+fileDisc+".");
           filename = "";
       }
       else if(!isValidFATName(filename))
       {
          errOut.println("Error: invalid "+fileDisc+": " +
                             filename);
          ++myCurrentLineError;
       }
       else
       {
          filename = filename.toUpperCase();
       }

       return filename;
   }

   /**
    * Returns aString padded with spaces or shorted to maxlen.
    * @param aString String the base string.
    * @param maxlen int the size of the string to return
    * @return aString padded with spaces or shorted to maxlen.
    */
   private String check_length(String aString, int maxlen)
   {
       StringBuffer rtnStr = new StringBuffer(aString);
       rtnStr.append("                                                      ");
       return rtnStr.substring(0,maxlen);
   }


   /**
    * @param filename String to check for valid characters.
    * @return true if all the characters in filename are in the valid set
    * of fat charaters.
    */
   private boolean isValidFATName(String filename)
   {
       String FATCHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_";
       filename = filename.trim();
       int len=filename.length();
       char aChar;
       for(int index=0;index<len;++index)
       {
           aChar = filename.charAt(index);
           if(FATCHARS.indexOf(aChar) < 0 )
           {
               return false;
           }
       }
       return true;
   }
}
/*
/ * This program converts a tab-delimited menu file into one suitable for
 * loading into the Cuttle Cart 2. * /

#include <stdio.h>
#include <iostream>
#include <string.h>
using namespace std;

/ *************************************************************************
 * Constants
 ************************************************************************* /

static const unsigned char CC2_HR = 0; / * horizontal rule character * /
static const size_t MAX_GAME_NAME_LEN = 40;
static const size_t MAX_FILE_NAME_LEN = 8;

/ *************************************************************************
 * Prototypes
 ************************************************************************* /

void makeupper(char *string);
void munge_filename(char *string);
int isValidFATName(const char *s);
void write_heading(FILE *outfile, char *string);
int convert_menu(const char *InputFilename, const char *OutputFilename);
int main(int argc, char **argv)
{
        if (argc < 3)
        {
                cerr << "Usage: convert_menu <inputfile> <outputfile>" << endl;
                return 1;
        }
        return convert_menu(argv[1], argv[2]);
}

void makeupper(char *string)
{
        size_t i;
        for(i = 0; i < strlen(string); i++)
        {
                string[i] = toupper(string[i]);
        }
}

void munge_filename(char *string)
{
        / * Allow a minus sign to mark file name fields without a value. * /
        if (0 == strcmp(string, "-"))
        {
                string[0] = 0;
        }
        else
        {
                makeupper(string);
        }
}

int isValidFATName(const char *s)
{
        static const char FATCHARS[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_";
        size_t first_bad_char = strspn(s, FATCHARS);
        return s[first_bad_char] == '\0';
}

/ * Write a title centered on a horizontal rule. * /
void write_heading(FILE *outfile, char *string)
{
        size_t i = 0;

        / * If the title is a single minus sign, display a horizontal rule with
         * no text.  Otherwise, leave room for 3 characters on each side of the
         * text (one space and two bits of line). * /
        if (0 != strcmp(string, "-"))
        {
                size_t textlen = strlen(string);
                size_t rulelen;

                / * Leave room for 3 chars on each side of name. * /
                if (textlen > MAX_GAME_NAME_LEN - 6)
                {
                        textlen = MAX_GAME_NAME_LEN - 6;
                }

                rulelen = MAX_GAME_NAME_LEN - textlen;

                while (i < (rulelen/2)-1)
                {
                        fputc(CC2_HR, outfile);
                        ++i;
                }

                fprintf(outfile, " %*.*s ", textlen, textlen, string);
                i += textlen + 2;
        }

        while (i < MAX_GAME_NAME_LEN)
        {
                fputc(CC2_HR, outfile);
                ++i;
        }
}

size_t fgettok(char *buf, size_t maxlen, const char *delim, FILE *f)
{
        size_t n = 0;
        int c;

        c = fgetc(f);
        if (c == EOF)
                goto done;

        while (n < maxlen)
        {
                if (strchr(delim, c))
                        break;

                buf[n++] = c;

                c = fgetc(f);
                if (c == EOF)
                        goto done;
        }

        if (n == maxlen) / * discard extra text characters * /
        {
                while (!strchr(delim, c))
                {
                        c = fgetc(f);
                        if (c == EOF)
                                goto done;
                }
        }

        while (strchr(delim, c)) / * discard delimiters * /
        {
                c = fgetc(f);
                if (c == EOF)
                        break;
        }

        ungetc(c, f);

 done:
        buf[n] = 0;
        return n;
}

int convert_menu(const char *InputFilename, const char *OutputFilename)
{
        const char seps[] = "\t\r\n";

        FILE *infile, *outfile;
        char gamename[MAX_GAME_NAME_LEN+1];
        char romname[MAX_FILE_NAME_LEN+1];
        char bankswitch[MAX_FILE_NAME_LEN+1];
        char startname[MAX_FILE_NAME_LEN+1];
        int result = 1;

        if((infile = fopen(InputFilename,"r")) == NULL)
        {
                cerr << "File " << InputFilename << " not found." << endl;
                goto done;
        }

        if((outfile = fopen(OutputFilename,"w")) == NULL)
        {
                cerr << "Unable to created file " << OutputFilename << "." << endl;
                goto close_input;
        }

        while (fgettok(gamename, MAX_GAME_NAME_LEN, seps, infile) > 0)
        {
                int error_count = 0;

                if (fgettok(romname, MAX_FILE_NAME_LEN, seps, infile) == 0)
                {
                        cerr << "Error: unexpected end of file" << endl;
                        goto close_files;
                }

                if (fgettok(bankswitch, MAX_FILE_NAME_LEN, seps, infile) == 0)
                {
                        cerr << "Error: unexpected end of file" << endl;
                        goto close_files;
                }

                if (fgettok(startname, MAX_FILE_NAME_LEN, seps, infile) == 0)
                {
                        cerr << "Error: unexpected end of file" << endl;
                        goto close_files;
                }

                munge_filename(romname);
                if (romname[0] && !isValidFATName(romname))
                {
                        cerr << "Error: invalid ROM file name: "
                             <<  romname << endl;
                        ++error_count;
                }

                munge_filename(bankswitch);
                if (bankswitch[0] && !isValidFATName(bankswitch))
                {
                        cerr << "Error: invalid bankswitch file name: "
                             <<  bankswitch << endl;
                        ++error_count;
                }

                munge_filename(startname);
                if (startname[0] && !isValidFATName(startname))
                {
                        cerr << "Error: invalid startup file name: "
                             <<  startname << endl;
                        ++error_count;
                }

                if (error_count)
                {
                        continue; / * discard this line and keep going * /
                isValidFATName}

                / * If all filenames are blank, display the game title as a
                 * full-width heading in a horizontal rule. * /
                if (!romname[0] && !bankswitch[0] && !startname[0])
                {
                        write_heading(outfile, gamename);
                }
                else
                {
                        fprintf(outfile, "%-*s",
                                MAX_GAME_NAME_LEN, gamename);
                }

                fprintf(outfile,"%-*s%-*s%-*s",
                        MAX_FILE_NAME_LEN, romname,
                        MAX_FILE_NAME_LEN, bankswitch,
                        MAX_FILE_NAME_LEN, startname);
        }

        result = 0;

 close_files:
        fclose(outfile);
 close_input:
        fclose(infile);
 done:
        return result;
}

*/
