package ccmenu;


import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;


/**
 * This Class looks in the given directory for known bin files and moves (and
 * renames them to 8.3 if needed) to the "GAMES" sub-directory and adds entries
 * for them to two text output files: MENU.CC2 and MENU.TXT
 *
 * <P>CC2ROMDB.TXT: This program uses a ROM database text file named
 * CC2ROMDB.TXT. Only ROM found in this database will be used. Example: 2 lines
 * from the database file:
 * <BR>7800: Ace Of Aces\t\t\t\tACE78\t\t78SC_LOW\t78BIOS\t\t04bbdc71
 * <BR>3D Tic-Tac-Toe\t\t\t\t\t3DTIC\t\t2K\t\t78BIOS\t\tb98f339e
 *
 * <P>MENU.TXT: This text menu file is for humans to update latter and create a
 * new MENU.CC2 from it using CC2MenuConvert. MENU.TXT has values separated by
 * one or more tab characters and speacal values defined only as "-" (a dash
 * character) will be treated as blank.
 * <BR>Example line from the file: "3D Tic-Tac-Toe\t\t\t3DTIC\t\t2K\t\t78BIOS\t"
 *
 * <P>MENU.CC2: This file is only defined by field length, and will not have any
 * carriage returns or line feeds.
 * <BR>Example one game and the start of another:
 * <br><pre>
 *  "3D Tic-Tac-Toe                          3DTIC   2K      78BIOS  A-Te"
 * <\pre>
 *
 *
 * <P> <b>The author assumes no responsibility for any damage caused by the use
 *  or posession of this program.  USE AT YOUR OWN RISK!!!<\b>
 * @copyright released to the public domain
 * @author Thomas R. Miller (converted to Java)
 *         Original C program by Chad Schell and Bob Colbert
 * @version 0.1 Intial release: 04/11/2004
 */
public class CC2MenuGenerate
{
   /**
    * The warning lines of text.
    */
   public static final String WARNING_STR =
      "This program will search the current directory for Atari ROMs\n" +
      "that it recognizes and move them to the subdirectory \"GAMES\" (which\n" +
      "it will create if it does not already exist.)  The ROMs will be renamed\n" +
      "into 8.3 format for use with the Cuttle Cart 2.  The original ROMs and any\n" +
      "duplicate ROMs will be deleted.\n" +
      "\nPlease backup your ROMs to another directory before proceeding.\n" +
      "Proceed?";

   /** Sub-directory to use to place the games found. */
   private static final String OUTPUT_ROMS_DIR = "GAMES";
   /** Text version of the cc2 menu output */
   private static final String OUTPUT_TXT_MENU = "MENU.TXT";
   /** cc2 menu output */
   private static final String OUTPUT_CC2_MENU = "MENU.CC2";


   private static final String DEFAULT_CONTROLLER = "Joystick";

   private static final String[][] CONTROLLS =
       {
         {"2", "2-Btn"},
         {"K", "Keypad"},
         {"P", "Paddles"},
         {"D", "Drive"},
         {"C", "CosmicCmd"},
         {"B", "BoostrGrp"},
         {"F", "FootCraz"},
         {"I", "KidCtrl"},
         {"V", "Joy+TPad"},
         {"L", "LGun"},
         {"JL", "Joy/LGun"},
         {"J2L", "2BJoy/LGun"},
         {"KV", "KidVid"},
         {"TB", "Trackball"}
       };

    /**
     * Default Constructor
     */
    public CC2MenuGenerate()
    {
    }


   /**
    * main: Starting place for the class.
    * @param args not used here
    */
   public static void main(String args[])
   {
      if(!printStart())
      {
         return;
      }
      CC2MenuGenerate menuGen = new CC2MenuGenerate();
      try
      {
         menuGen.generateMenu(new File("."), true, false, false, null);
      }
      catch(Exception ex)
      {
         System.out.println("Error in processing:" + ex.getMessage() +
                            "\nExiting");
         System.err.print(ex);
         ex.printStackTrace(System.err);
      }
   }


   /**
    * This method looks in the romDir directory for known bin files and moves
    * them to the GAMES sub-directory and adds entries for them
    * to two text output files: MENU.CC2 and MENU.TXT
    * @param romDir the directory to look for roms, also will add/use this
    * directory to place the sub-directory GAMES in.
    * @param addByName boolean
    * @param addByCo boolean
    * @param addByController boolean
    * @param aProgressDlg ProgressDlg
    * @throws Exception
    * @return int
    */
   public int generateMenu(File romDir,
                            boolean addByName,
                            boolean addByCo,
                            boolean addByController,
                            ProgressDlg aProgressDlg)
    throws Exception
   {
      int rtnNumRoms = 0;
      CC2Database newDB = new CC2Database();

      rtnNumRoms = moveFiles(newDB, romDir, aProgressDlg);

      if(rtnNumRoms > 0)
      {
         createMenuFiles(newDB, romDir, addByName, addByCo, addByController);
      }
      return rtnNumRoms;
   }


   /**
    * Finds rom files listed in the database and moves them to the games dir.
    *
    * Calculates CRC's of known Atari rom file and if the CRC is in the
    * database moves them to the GAMES directory and updates the record to
    * indecate that it was found.
    *
    * @param database the rom database to use
    * @param romDir File of directory to look for Atari rom files
    * @param progressDlg ProgressDlg
    * @throws Exception on error processing
    * @return number of rom files
    */
   private int moveFiles(CC2Database database, File romDir,
                                ProgressDlg progressDlg)
           throws Exception
   {
      File currBin;
      File outRomDir = new File(romDir, OUTPUT_ROMS_DIR);
      FileHelper theFileHelper = FileHelper.getInstance();
      String[] binFiles = romDir.list(new RomFileFilter());
      RomEntry aRomEntry;
      int totalRoms;
      int inMenu = 0;
      int unknown = 0;
      int duplicate = 0;
      int totalProcessed = 0;

      if (!outRomDir.exists())
      {
         theFileHelper.createFilePath(outRomDir);
      }
      else if(!outRomDir.isDirectory())
      {
          throw new Exception("GAMES is not a directory.");
      }
      if(null == binFiles)
      {
         binFiles = new String[0];
      }

      totalRoms = binFiles.length;
      progressDlg.setMaximum(binFiles.length);

      for (int index = 0; index < binFiles.length; ++index)
      {
         currBin = new File(romDir, binFiles[index]);
         // compute bin file's CRC
         aRomEntry = database.lookup(theFileHelper.calcCRC(currBin));
         if(null != aRomEntry && !aRomEntry.isIncluded())
         {
            if (aRomEntry.needsCC2Fix())
            {
               theFileHelper.fix7800Rom(currBin);
               aRomEntry = database.lookup(theFileHelper.calcCRC(currBin));

               if (null == aRomEntry)
               {
                  System.err.println("Error fixing file:" +
                                     currBin.getAbsolutePath());
                  ++totalProcessed;
                  ++unknown;
                  continue;
               }
            }
            // move file & rename
            File toFile = new File(outRomDir, aRomEntry.getRomFileName());
            theFileHelper.moveFile(currBin, toFile, true);

            aRomEntry.setIncluded();
            ++totalProcessed;
            ++inMenu;
            progressDlg.setValue(totalProcessed, inMenu, unknown, duplicate);
         }
         else
         {
            if(null != aRomEntry)
            {
               ++unknown;
            }
            else
            {
               ++duplicate;
            }
            ++totalProcessed;
            progressDlg.setValue(totalProcessed, inMenu, unknown, duplicate);
         }
      }

      return totalRoms;
   }


   /**
    * Looks for found rom files in the database and puts an entry for each one
    * in to both a text menu file and a cc2 menu file.
    * @param database the CC2Database to use
    * @param txtDir txtDir a File to put the new Menu files
    * @param addByName boolean
    * @param addByCo boolean
    * @param addByController boolean
    */
   private void createMenuFiles(CC2Database database, File txtDir,
                                boolean addByName,
                                boolean addByCo,
                                boolean addByController)
   {
      File txtMenuFile = new File(txtDir, OUTPUT_TXT_MENU);
      File cc2MenuFile = new File(txtDir, OUTPUT_CC2_MENU);
      BufferedWriter txtOutput = null;
      BufferedWriter cc2Output = null;

      // now create output files:
      try
      {
         txtOutput = new BufferedWriter(new FileWriter(txtMenuFile));
         cc2Output = new BufferedWriter(new FileWriter(cc2MenuFile));

         if(addByName)
         {
            addMenu_NameOrder(database, txtOutput, cc2Output);
         }
         if(addByCo)
         {
            addMenu_companyOrder(database, txtOutput, cc2Output);
         }
         if(addByController)
         {
            addMenu_controllerOrder(database, txtOutput, cc2Output);
         }
      }
      catch (IOException ex)
      {
         System.err.println("Can't write menu files.");
         System.err.println(ex.getMessage());
         return;
      }
      finally
      {
          try
          {
            // close files
            cc2Output.close();
            txtOutput.close();
          }
          catch (IOException ex1)
          {
          }
      }
   }


   /**
    * Adds menu entries in game name order.
    * @param database the CC2Database with entries added.s
    * @param txtOutput the text file BufferedWriter to output entries to.
    * @param cc2Output the cc2 menu file BufferedWriter to output entries to.
    * @throws IOException if this has a problem writing the entries.
    */
   private void addMenu_NameOrder(CC2Database database,
                                   BufferedWriter txtOutput,
                                   BufferedWriter cc2Output)
    throws IOException
   {
      Iterator includeRomsIterator = database.getAllIncluded();
      RomEntry aRomEntry;

      // loop though the rom entries and write them out
      while(includeRomsIterator.hasNext())
      {
         aRomEntry = (RomEntry) includeRomsIterator.next();
         // add entry to MENU.CC2
         cc2Output.write(aRomEntry.getCC2MenuEntry());
         // add entry to MENU.TXT
         txtOutput.write(aRomEntry.getTxtMenuEntry() + "\n");
      }
   }


   /**
    *
    * @param database CC2Database
    * @param txtOutput BufferedWriter
    * @param cc2Output BufferedWriter
    * @throws IOException
    */
   private void addMenu_companyOrder(CC2Database database,
                                   BufferedWriter txtOutput,
                                   BufferedWriter cc2Output)
    throws IOException
   {
      Iterator includeRomsIterator = database.getAllIncluded();
      RomEntry aRomEntry;
      MultiHash companyHash = new MultiHash();
      String company = null;
      List aList = null;

      // loop though the rom entries and write them out
      while(includeRomsIterator.hasNext())
      {
         aRomEntry = (RomEntry) includeRomsIterator.next();
         // add to list of company:
         company = aRomEntry.getCompany();
         if(null == company)
         {
           company = "UNKNOWN";
         }
         if(null != aRomEntry && null != company)
         {
           companyHash.put(aRomEntry.getCompany(), aRomEntry);
         }
      }

      aList = new ArrayList(companyHash.keySet());

      // sort
      java.util.Collections.sort(aList);

      Iterator compItr = aList.iterator();
      while(compItr.hasNext())
      {
        company = (String) compItr.next();

        // add seperator
        aRomEntry = (RomEntry) RomEntry.LINE_ENTRY.clone();
        aRomEntry.setName(centerString(company));
        // add entry to MENU.CC2
        cc2Output.write(aRomEntry.getCC2MenuEntry());
        // add entry to MENU.TXT
        txtOutput.write(aRomEntry.getTxtMenuEntry() + "\n");


        aList = new ArrayList(companyHash.get(company));

        // sort
        java.util.Collections.sort(aList);


        includeRomsIterator = aList.iterator();
        while(includeRomsIterator.hasNext())
        {
           aRomEntry = (RomEntry) includeRomsIterator.next();
              // add entry to MENU.CC2
              cc2Output.write(aRomEntry.getCC2MenuEntry());
              // add entry to MENU.TXT
              txtOutput.write(aRomEntry.getTxtMenuEntry() + "\n");
        }
      }
      // add a line entry
      cc2Output.write(RomEntry.LINE_ENTRY.getCC2MenuEntry());
      // add entry to MENU.TXT
      txtOutput.write(RomEntry.LINE_ENTRY.getTxtMenuEntry() + "\n");
   }


   /**
    *
    * @param database CC2Database
    * @param txtOutput BufferedWriter
    * @param cc2Output BufferedWriter
    * @throws IOException
    */
   private void addMenu_controllerOrder(CC2Database database,
                                   BufferedWriter txtOutput,
                                   BufferedWriter cc2Output)
    throws IOException
   {
      Iterator includeRomsIterator = database.getAllIncluded();
      RomEntry aRomEntry;
      MultiHash controllerHash = new MultiHash();
      String ctrKey = null;
      String ctlString = null;
      List aList = null;
      HashMap controlls = new HashMap(CONTROLLS.length);
      for(int index=0; index < CONTROLLS.length; ++index)
      {
         controlls.put(CONTROLLS[index][0], CONTROLLS[index][1]);
      }


      // loop though the rom entries and write them out
      while(includeRomsIterator.hasNext())
      {
         aRomEntry = (RomEntry) includeRomsIterator.next();
         // add to list of company:
         ctrKey = aRomEntry.getController();
         if(null == ctrKey || ctrKey.trim().length() < 1)
         {
            ctlString = DEFAULT_CONTROLLER;
         }
         else
         {
            ctlString = (String) controlls.get(ctrKey);
         }
         controllerHash.put(ctlString, aRomEntry);
      }

      aList = new ArrayList(controllerHash.keySet());

      // sort
      java.util.Collections.sort(aList);

      Iterator compItr = aList.iterator();
      while(compItr.hasNext())
      {
        ctlString = (String) compItr.next();

        // add seperator
        aRomEntry = (RomEntry) RomEntry.LINE_ENTRY.clone();
        aRomEntry.setName(centerString(ctlString));
        // add entry to MENU.CC2
        cc2Output.write(aRomEntry.getCC2MenuEntry());
        // add entry to MENU.TXT
        txtOutput.write(aRomEntry.getTxtMenuEntry() + "\n");


        aList = new ArrayList(controllerHash.get(ctlString));

        // sort
        java.util.Collections.sort(aList);


        includeRomsIterator = aList.iterator();
        while(includeRomsIterator.hasNext())
        {
           aRomEntry = (RomEntry) includeRomsIterator.next();
           // add entry to MENU.CC2
           cc2Output.write(aRomEntry.getCC2MenuEntry());
           // add entry to MENU.TXT
           txtOutput.write(aRomEntry.getTxtMenuEntry() + "\n");
        }
      }
      // add a line entry
      cc2Output.write(RomEntry.LINE_ENTRY.getCC2MenuEntry());
      // add entry to MENU.TXT
      txtOutput.write(RomEntry.LINE_ENTRY.getTxtMenuEntry() + "\n");
   }


   /**
    *
    * @param aStr String
    * @return String
    */
   private String centerString(String aStr)
   {
     String dashes = "----------------------------------------";
     StringBuffer rtnStr = new StringBuffer();
     int dashSize = (40 - aStr.length())/2;
     String dashToAdd = dashes.substring(0,dashSize);
     rtnStr.append(dashToAdd);
     rtnStr.append(aStr);
     rtnStr.append(dashToAdd);
     int size = rtnStr.length();
     if(size > 40)
     {
       rtnStr.delete(size - 40, size);
     }
     size = rtnStr.length();
     if(size < 40)
     {
       rtnStr.append(dashes);
       size = rtnStr.length();
       rtnStr.delete(40, size);
     }

     return rtnStr.toString();
   }


   /**
    * Prints the user warning message and asks if User want to proceed.
    * @return boolean of user's decision to proceed.
    */
   static private boolean printStart()
   {
      System.out.println(WARNING_STR + " (Y/N): ");
      try
      {
         int inChar = System.in.read();
         return (inChar == 'Y' || inChar == 'y');
      }
      catch (IOException ex)
      {
         System.err.println("can't read command line");
         return false;
      }
   }
}
