package ccmenu;

import java.io.*;
import java.util.ArrayList;
//import java.util.*;
import java.util.zip.CRC32;


/**
 * Singleton class used to help with file operations.
 */
public class FileHelper
{
    /** CRC-32 polynomial reversed */
    private static final long CRC32_REV = 0xA0000001L;

    /** This is the size of the crc lookup table */
    private static final int CRC_TABLE_SIZE = 256;

    /** Number to add to a negative byte to make positive */
    private static final int TWOS_COMP_BYTE = 0x0100;

    /** This is a crc lookup table */
    private long[] myCrcTable = new long[CRC_TABLE_SIZE];

    /** The maximum file size alowed (512*1024)*/
    private static final int MAX_FILE_SIZE = 0x80000;

    /** A buffer to be used to read in files */
    private byte[] myBuffer = new byte[ MAX_FILE_SIZE ];

    /** Singleton instance of this class */
    private static FileHelper ourInstance = new FileHelper();

    /**
     * Constructer: private to inforce the singleton concept.
     * @see #getInstance
     * Sets up the crc lookup table.
     */
    private FileHelper()
    {
       for(int index=0; index < CRC_TABLE_SIZE; ++index)
       {
          myCrcTable[index] = crcrevhware(index, CRC32_REV, 0);
       }
    }


    /**
     * @return FileHelper the singleton instance of this class.
     */
    public static FileHelper getInstance()
    {
       return ourInstance;
    }


    /**
     * Calculate the CRC for this file.
     * @param aFile the file to calculate
     * @return String of the CRC value.
     */
    public String calcCRC(File aFile)
    throws Exception
    {
        long crc = 0;
        final int fileSize = (int) aFile.length();
        DataInputStream inStream = null;

        // create a new CRC-calculating object
        //CRC32 crc2 = new CRC32(); This would work for CRC32 but
        //                        this is different than the one in CC2ROMDB.TXT
        //crc2.update(myBuffer, 0, fileSize);
        //crc = crc2.getValue();

        if(fileSize > MAX_FILE_SIZE)
        {
            throw new Exception("Error:  File " + aFile.getAbsolutePath() +
                                              " is too large, skipping.");


        }

        try
        {
           inStream = new DataInputStream(new FileInputStream(aFile));
           // *** READ ***
           if( fileSize != inStream.read(myBuffer))
           {
              throw new Exception("Error:  File " + aFile.getAbsolutePath() +
                                                " is too large, skipping.");
           }
           for(int index=0; index < fileSize; ++index)
           {
              crc = ((crc >> 8) ^
                          myCrcTable[(int)((crc ^ myBuffer[index]) & 0xff)])
                          & 0xffffffff;
           }
        }
        catch (IOException iex)
        {
             System.err.println("Error reading file:"+aFile.getAbsolutePath());
             return "";
        }
        finally
        {
            if(null != inStream)
            {
                try
                {
                   inStream.close();
                }
                catch (IOException iex)
                {
                }
            }
        }

        String rtnString = "000000" +Long.toHexString(crc);
        return rtnString.substring(rtnString.length() - 8);
    }



    /**
     * Calculates one crcTable entry
     * @param data the number entry to calc
     * @param genpoly a number used
     * @param accum number to start with
     * @return long of the crcTable entry
     */
    private static long crcrevhware(long data, long genpoly, long accum)
    {
        data <<= 1;
        for(int index=8; index > 0; --index)
        {
            data >>= 1;
            if(((data ^ accum) & 1) != 0)
                accum = (accum >> 1) ^ genpoly;
            else
                accum >>= 1;
        }
        return(accum);
    }


    /**
     * Moves file from orgFile to destFile
     * This method may fail on some platforms when moving accross folums.
     * @param orgFile the file to move
     * @param destFile the destination file
     * @param overwrite overwrites the destFile if it exists
     * otherwhise throws an Exception if destFile exists
     * @throws Exception if error moving file.
     */
    public void moveFile(File orgFile, File destFile, boolean overwrite)
            throws Exception
    {
        if(orgFile.equals(destFile))
        {
           throw new Exception("Can not move file:" + orgFile +
                               " to " + destFile +
                               " because files are the same.");
        }

        //make sure file didn't already exist
        if(destFile.exists())
        {
           if (!overwrite)
           {
              throw new Exception("Can not move file:" + orgFile +
                                  " to " + destFile +
                                  " because destination already exists.");
           }
           else
           {
              //delete since rename won't overwrite files
              destFile.delete();
           }
        }

       // move the file from
       if(!orgFile.renameTo(destFile))
       {
          throw new Exception("Can not move file:" + orgFile +
                              " to " + destFile);
       }
    }


    /**
     * This method creates the directories for the file to be created if they
     * do not exist.
     * @param directory The directory to be created.
     * @throws Exception if the File in directory is a directory.
     */
    public void createFilePath(File directory)
            throws Exception
    {
       if(!directory.exists())
       {
          File theDirectory = directory;
          ArrayList dirList = new ArrayList();

          // While the directory does not exist and haven't reached the
          // beginning of the path.  Add this directory to the list of
          // directories and get the parent of this directory
          while(theDirectory != null && !theDirectory.exists())
          {
             dirList.add(theDirectory);
             theDirectory = theDirectory.getParentFile();
          }

          // Loop through the list of directories in reverse order (since they
          // were added to the list with subdirectories first) and create the
          // directories
          File dir = null;
          for(int dirIndex = dirList.size()-1; dirIndex >= 0; --dirIndex)
          {
             dir = (File)dirList.get(dirIndex);
             dir.mkdir();
          }
       }
       else if(!directory.isDirectory())
       {
           throw new Exception("Can not create Directory:" +
                               directory.getAbsolutePath() +
                               " because there is a file with same name. ");
       }
    }


    /**
     * Moves the extra 16K bank from the start to the end of the ROM file.
     * 7800 Crossbow and Alien Brigade do not work
     * as they are most commonly distributed (on the CC2).
     * @param romFile pointer to the rom file that needs fixing.
     */
    public void fix7800Rom(File romFile)
    {
        DataInputStream inStream = null;
        DataOutputStream outStream = null;
        File tempfile = null;
        byte[] buffer = new byte[144 * 1024];
        try
        {
            tempfile = File.createTempFile("cc2", "tmp", romFile.getParentFile());
            inStream = new DataInputStream(new FileInputStream(romFile));
            outStream = new DataOutputStream(new FileOutputStream(tempfile));
            inStream.read(buffer,128 * 1024, 16 * 1024);
            inStream.read(buffer,0, 128 * 1024);
            outStream.write(buffer);
            inStream.close();
            outStream.close();

            // delete first file.
            romFile.delete();
            // rename temp to first
            tempfile.renameTo(romFile);
        }
        catch(IOException x)
        {
            System.err.println("Error reading file:"+romFile.getAbsolutePath());
            return;
        }
        return;
    }

    public static void main(String[] args)
    {
       File theFile = new File("rescue.a78");
       if(theFile.exists())
       {
          getInstance().fix7800Rom(theFile);
       }
       else
       {
          System.err.println("not exist" + theFile.getAbsolutePath());
       }
    }

}
