package ccmenu;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * This class enables the client to hold more than one value related to the
 * same key. The values are held in a List and can be retrieved as such
 * when the client looks up the key.
 */
public class MultiHash
{

   /** By default, this data structure is not thread safe */
   private static final boolean DEFAULT_THREAD_SAFETY = false;


   /** The internal Map that holds the key->bucket pairs */
   private Map myBuckets = null;


   /**
    * Default constructor. Initializes the bucket hash
    */
   public MultiHash()
   {
      this(DEFAULT_THREAD_SAFETY);
   }


   /**
    * Constructor. Initializes the bucket hash based on whether or not thread
    * safety is necessary
    * @param requireThreadSafety Indicates whether or not this data structure
    * should be constructed to be thread safe
    */
   public MultiHash(boolean requireThreadSafety)
   {
      if (requireThreadSafety)
      {
         myBuckets = new Hashtable();
      }
      else
      {
         myBuckets = new HashMap();
      }
   }


   /**
    * Put a new value into the bucket associated with the specified key
    * @param key The key that maps to the bucket where the specified value
    * will be stored
    * @param value The value to insert into the appropriate bucket
    */
   public void put(Object key, Object value)
   {
      List aBucket = null;
      if (myBuckets.containsKey(key))
      {
         aBucket = (List)myBuckets.get(key);
      }
      else
      {
         aBucket = new ArrayList();
      }
      aBucket.add(value);
      myBuckets.put(key, aBucket);
   }


   /**
    * Get all values in the bucket
    * @param key The key identifying the bucket to retrieve
    * @return A List representing the bucket with the specified key
    */
   public List get(Object key)
   {
      return (List)myBuckets.get(key);
   }


   /**
    * Check to see whether the bucket hash contains the specified key
    * @param key The key to check
    * @return True if the bucket hash contains the specified key, false
    * otherwise
    */
   public boolean containsKey(Object key)
   {
      return myBuckets.containsKey(key);
   }


   /**
    * Check to see whether the bucket hash contains the specified key/value pair
    * @param key The key to check
    * @param value The value to check
    * @return True if the bucket hash contains the specified key/value, false
    * otherwise
    */
   public boolean containsValue(Object key, Object value)
   {
      boolean returnValue = false;
      List aList = get(key);

      if (aList != null)
      {
         returnValue = aList.contains(value);
      }

      return returnValue;
   }


   /**
    * Get the value with the specified key
    * @param key The key of the value to retrieve
    * @param value The value to retrieve
    * @return The specified value
    */
   public Object getValue(Object key, Object value)
   {
      Object returnValue = null;
      List aList = get(key);

      if (aList != null)
      {
         returnValue = aList.get(aList.indexOf(value));
      }

      return returnValue;
   }


   /**
    * Remove the entire bucket associated with the specified key
    * @param key The key whose bucket will be removed. The key will also be
    * removed from the hash
    * @return The bucket being removed
    */
   public List remove(Object key)
   {
      return (List)myBuckets.remove(key);
   }


   /**
    * Remove a given value from the Bucket with the specified key
    * @param key The key of the bucket from which to remove the specified value
    * @param objToRemove The value to remove
    */
   public void remove(Object key, Object objToRemove)
   {
      List aList = get(key);

      if (aList != null)
      {
         aList.remove(objToRemove);
         if (aList.isEmpty())
         {
            remove(key);
         }
      }
   }


   /**
    * Retrieve the entire key set held within this bucket hash
    * @return The Set of keys held within the bucket hash
    */
   public Set keySet()
   {
      return myBuckets.keySet();
   }


   /**
    * Clear this BucketHash
    */
   public void clear()
   {
      Iterator iBuckets = myBuckets.values().iterator();
      List currentList = null;

      while (iBuckets.hasNext())
      {
         currentList = (List)iBuckets.next();
         currentList.clear();
         currentList = null;
      }

      myBuckets.clear();
   }
}
