package ccmenu;


/**s
 * Defaines a data class to store information about a specific roms.
 */
public class RomEntry
   implements Comparable
{
   /** String used to define a blank/emty value string */
   private static final String EMTY_VALUE = "-";

   /** a blank Entry line to use */
   public static final RomEntry LINE_ENTRY =
       new RomEntry("----------------------------------------",
                    EMTY_VALUE,EMTY_VALUE,EMTY_VALUE,EMTY_VALUE);

   /** The extension to use for rom files*/
   private static final String CC2_ROM_EXT = ".BIN";
   /** The valid FAT chars to use */
   private static String FATCHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_";

   /**
    * 7800 Crossbow and Alien Brigade do not work as they are most
    * commonly distributed.  The extra 16K bank needs to be moved from the
    * start to the end of the ROM
    */
   private static final String[] NEED_FIX_ROMS = {"7369E9F6", "14DE7304"};

   /** The menu display name */
   private String myName;
   /** The File to use in the Game directory */
   private String myRomName;
   /** The backswitching sceam to use */
   private String myBankswitch;
   /** The startup sceam to use */
   private String myStartup;
   /** This Rom's CRC to use, the primary key */
   private String myCRC;
   /** The company name */
   private String myCompany;
   /** The Rarity of this cartrige, I don't know about this one! */
   private String myRarity;
   /** The controller used */
   private String myController;
   /** This has been found and in the GAMES directory. */
   private boolean myIncluded = false;


   /* Indexes of strings to use */
   private static final int DB_GAMENAME   = 0;
   private static final int DB_ROMNAME    = 1;
   private static final int DB_BANKSWITCH = 2;
   private static final int DB_STARTUP    = 3;
   private static final int DB_CRC        = 4;
   private static final int DB_COMPANY    = 5;
   private static final int DB_RARITY     = 6;
   private static final int DB_CONTROLLER = 7;

   /** Field names for display uses, currently not used */
   private static final String[] DB_FIELD_DISC = {"Game Name","ROM Name",
             "Bankswitch file name", "Startup file name", "CRC number",
             "Company", "Rarity", "Controller"};

   /** The max length of a game name/discription */
   static final int MAX_GAME_NAME_LEN = 40;

   /** The max length of a FILE name */
   static final int MAX_FILE_NAME_LEN = 8;

   /** MENU.CC2 entry length */
   public static final int MENU_CHAR_LEN =
       MAX_GAME_NAME_LEN + 3 * MAX_FILE_NAME_LEN;

   /** Length of a Tab */
   static final int TAB_WIDTH = 8;

   /** used to seperate fields in text output */
   final static char TXT_SEP = '\t';


   /**
    * Constructer
    * @param name String
    * @param romName String
    * @param bankswitch String
    * @param startup String
    * @param crc String
    * @param company String
    * @param rarity String
    */
   public RomEntry(String name,
                   String romName,
                   String bankswitch,
                   String startup,
                   String crc,
                   String company,
                   String rarity)
   {
      this(name, romName, bankswitch, startup, crc);
      myCompany = company;
      myRarity= rarity;
   }


   /**
    * Constructer
    * @param name String
    * @param romName String
    * @param bankswitch String
    * @param startup String
    * @param crc String
    */
   public RomEntry(String name,
                   String romName,
                   String bankswitch,
                   String startup,
                   String crc)
   {
      myName = name;
      myRomName = romName;
      myBankswitch = bankswitch;
      myStartup = startup;
      myCRC = crc;
      checkValid();
   }

   /**
    * Constructer using a database line with tab characters.
    * @param aDBLine String
    */
   public RomEntry(String aDBLine)
   {
      String[]  entries = aDBLine.split("\\t+"); // split on one or more tabs
      if (entries.length == DB_CRC + 1)
      {
        myName = entries[DB_GAMENAME];
        myRomName = entries[DB_ROMNAME];
        myBankswitch = entries[DB_BANKSWITCH];
        myStartup = entries[DB_STARTUP];
        myCRC = entries[DB_CRC];
      }
      else if (entries.length == DB_RARITY + 1)
      {
        myName = entries[DB_GAMENAME];
        myRomName = entries[DB_ROMNAME];
        myBankswitch = entries[DB_BANKSWITCH];
        myStartup = entries[DB_STARTUP];
        myCRC = entries[DB_CRC];
        myCompany = entries[DB_COMPANY];
        myRarity= entries[DB_RARITY];
      }
      else if (entries.length == DB_CONTROLLER + 1)
      {
        myName = entries[DB_GAMENAME];
        myRomName = entries[DB_ROMNAME];
        myBankswitch = entries[DB_BANKSWITCH];
        myStartup = entries[DB_STARTUP];
        myCRC = entries[DB_CRC];
        myCompany = entries[DB_COMPANY];
        myRarity= entries[DB_RARITY];
        myController = entries[DB_CONTROLLER];
      }
      checkValid();
   }


   /**
    * Takes the correct number of
    * @param aCC2MenuEntry char[]
    * @throws Exception
    */
   public RomEntry(char[] aCC2MenuEntry)
      throws Exception
   {
     if(aCC2MenuEntry.length < MENU_CHAR_LEN)
     {
       throw new Exception("Incorrect number of bytes in CC2 Menu entry.");
     }
     // read in each field:
     String entryStr = String.valueOf(aCC2MenuEntry);
     int currentPos = 0;

     myName = entryStr.substring(currentPos, currentPos+MAX_GAME_NAME_LEN);
     currentPos += MAX_GAME_NAME_LEN;
     myRomName = entryStr.substring(currentPos, currentPos+MAX_FILE_NAME_LEN);
     currentPos += MAX_FILE_NAME_LEN;
     myBankswitch = entryStr.substring(currentPos, currentPos+MAX_FILE_NAME_LEN);
     currentPos += MAX_FILE_NAME_LEN;
     myStartup = entryStr.substring(currentPos, currentPos+MAX_FILE_NAME_LEN);
   }


   /**
    *
    * @return boolean
    */
   public boolean needsCC2Fix()
   {
     boolean rtnVal=false;
     for(int index=0;index < NEED_FIX_ROMS.length; ++index)
     {
        rtnVal = rtnVal || myCRC.equalsIgnoreCase(NEED_FIX_ROMS[index]);
     }
     return rtnVal;
   }

   /**
    * @return The name of the file to put in the GAMES directory.
    */
   public String getRomFileName()
   {
      return myRomName.trim() + CC2_ROM_EXT;
   }


   /**
    * @return Menu entry name
    */
   public String getName()
   {
     return myName;
   }


   /**
    * @param newName Set the Menu entry name
    */
   public void setName(String newName)
   {
     myName = check_length(newName, MAX_GAME_NAME_LEN);
   }


   /**
    * Set the company name
    * @param company the new company more.
    */
   public void setCompany(String company)
   {
      myCompany = company;
   }

   /**
    * @return the company name
    */
   public String getCompany()
   {
     return myCompany;
   }

   /**
    * Set the rarity
    * @param rarity the rarity String
    */
   public void setRarity(String rarity)
   {
      myRarity = rarity;
   }

   /** @return get the rarity String */
   public String getRarity()
   {
     return myRarity;
   }

   /**
    * Set the new CRC
    * @param newCRC String
    */
   public void setCRC(String newCRC)
   {
     myCRC = newCRC;
   }


   /**
    * @return the CRC String
    */
   public String getCRC()
   {
      return myCRC;
   }


   /**
    * Sets to this new controller.
    * @param controller new controller
    */
   public void setController(String controller)
   {
      myController = controller;
   }


   /**
    * @return return controller String
    */
   public String getController()
   {
     return myController;
   }


   /**
    * Returns a string for the MENU.CC2 file.
    * <P>MENU.CC2: This file is only defined by field length, and will not have
    * any carriage returns or line feeds.
    * <BR>Example one game and the start of  another:
    * "3D Tic-Tac-Toe                          3DTIC   2K      78BIOS  A-Te"
    * @return the text entry.
    */
   public String getCC2MenuEntry()
   {
      StringBuffer rtnVal = new StringBuffer(myName);
      rtnVal.append(myRomName);
      rtnVal.append(myBankswitch);
      rtnVal.append(myStartup);
      return rtnVal.toString();
   }

   /**
    * @return a String for the TXT Menu file.
    */
   public String getTxtMenuEntry()
   {
      StringBuffer rtnVal = new StringBuffer();
      rtnVal.append(pretty_print_field(myName.trim(), MAX_GAME_NAME_LEN));
      rtnVal.append(TXT_SEP);
      rtnVal.append(pretty_print_field(myRomName.trim(), MAX_FILE_NAME_LEN));
      rtnVal.append(TXT_SEP);
      rtnVal.append(pretty_print_field(myBankswitch.trim(), MAX_FILE_NAME_LEN));
      rtnVal.append(TXT_SEP);
      rtnVal.append(pretty_print_field(myStartup.trim(), MAX_FILE_NAME_LEN));
      return rtnVal.toString();
   }


   /**
    * Returns a string for the MENU.CC2 file.
    * <P>MENU.CC2: This file is only defined by field length, and will not have
    * any carriage returns or line feeds.
    * <BR>Example one game and the start of  another:
    * "3D Tic-Tac-Toe                          3DTIC   2K      78BIOS  A-Te"
    * @return String
    */
   public String getDBEntry()
   {
     StringBuffer rtnVal = new StringBuffer();
     rtnVal.append(pretty_print_field(myName.trim(), MAX_GAME_NAME_LEN));
     rtnVal.append(TXT_SEP);
     rtnVal.append(pretty_print_field(myRomName.trim(), MAX_FILE_NAME_LEN));
     rtnVal.append(TXT_SEP);
     rtnVal.append(pretty_print_field(myBankswitch.trim(), MAX_FILE_NAME_LEN));
     rtnVal.append(TXT_SEP);
     rtnVal.append(pretty_print_field(myStartup.trim(), MAX_FILE_NAME_LEN));
     rtnVal.append(TXT_SEP);
     rtnVal.append(myCRC.trim());
     rtnVal.append(TXT_SEP);
     if(null != myCompany)
     {
        rtnVal.append(myCompany.trim());
     }
     rtnVal.append(TXT_SEP);
     if(null != myRarity)
     {
        rtnVal.append(myRarity.trim());
     }
     rtnVal.append(TXT_SEP);
     if(null != myController)
     {
        rtnVal.append(myController.trim());
     }

     return rtnVal.toString();
   }


   /**
    * Returns one field with the correct amount of tab charaters at the end for
    * the MENU.TXT file.
    * @param s the field String
    * @param field_width length of the field
    * @return a String with the correct amount of tab charaters
    */
   private String pretty_print_field(String s, int field_width)
   {
      StringBuffer rtnStr = new StringBuffer();
      if (null != s)
      {
         int len = s.trim().length();

         if (len > 0)
         {
            rtnStr.append(s);

            for (len = s.length();
                 len < field_width;
                 len = (len - (len % TAB_WIDTH)) + TAB_WIDTH)
            {
               rtnStr.append('\t');
            }

            return rtnStr.toString();
         }
      }
      rtnStr.append("-\t");
      return rtnStr.toString();
   }


   /**
    * Will set each field to the correct length.
    */
   private void checkValid()
   {
      myName = check_length(myName, MAX_GAME_NAME_LEN);
      myRomName = check_length(myRomName, MAX_FILE_NAME_LEN);
      myBankswitch = check_length(myBankswitch, MAX_FILE_NAME_LEN);
      myStartup = check_length(myStartup, MAX_FILE_NAME_LEN);
      myCRC = check_length(myCRC, MAX_FILE_NAME_LEN);
   }

   /**
    * Returns a String with maxlen length.  If aString is too small, will add
    * blanks; if aString is too big will trim off.
    * @param aString starting String
    * @param maxlen the length of the String to return.
    * @return a String with maxlen length.
    */
   private String check_length(String aString, int maxlen)
   {
      if(null == aString)
      {
         System.err.println("error: input string is null.");
         aString = "";
      }
      else if(aString.trim().equals(EMTY_VALUE))
      {
         aString = "";
      }
      StringBuffer rtnStr = new StringBuffer(aString);
      rtnStr.append("                                                      ");
      return rtnStr.substring(0, maxlen);
   }


   /**
    * Makes sure all chars in filename are in the list of valid char set.
    * @param filename String to check
    * @return boolean true if all chars are in the valid char set.
    */
   private boolean isValidFATName(String filename)
   {
      filename = filename.trim();
      int len = filename.length();
      char aChar;
      for (int index = 0; index < len; ++index)
      {
         aChar = filename.charAt(index);
         if (FATCHARS.indexOf(aChar) < 0)
         {
            return false;
         }
      }
      return true;
   }


   /**
    * Sets this entry to be included.
    */
   public void setIncluded()
   {
      myIncluded = true;
   }


   /**
    * @return boolean true if this entry has been set to included.
    */
   public boolean isIncluded()
   {
      return myIncluded;
   }


   /**
    * An entry should appear in the menu if it has a ROM
    * file and bankswitch and startup files.  Entries
    * that have a ROM file but no bankswitch or startup
    * files represent Supercharger loads that do not need
    * their own menu entries.
    * @return boolean true if this rom is to be listed.
    */
   public boolean isListed()
   {
       return myRomName.trim().length() > 0 &&
              myBankswitch.trim().length() > 0 &&
              myStartup.trim().length() > 0;
   }


   /**
    * Used for comparing two RomEntry objects
    * @param obj RomEntry object to compare to
    * @return int 0 if equal, < 0 if this comes before obj, > 0 comes after obj
    */
   public int compareTo(Object obj)
   {
      RomEntry rightRow = (RomEntry) obj;
      return myName.compareTo(rightRow.myName);
   }


   /**
    * Clone this RomEntry.
    * @return a new RomEntry with the same information as this one.
    */
   public Object clone()
   {
         return new RomEntry(myName, myRomName, myBankswitch, myStartup, myCRC,
                             myCompany, myRarity);
   }
}
