/* Fenix - Compilador/intrprete de videojuegos
 * Copyright (C) 1999 Jos Luis Cebrin Page
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "fxc.h"
#include "messages.c"

/* ---------------------------------------------------------------------- */
/* Tokenizador. La funcin token_next, ampliamente utilizada, recoge el   */
/* siguiente token (identificador, operador, etc) del cdigo fuente, y    */
/* rellena la estructura global "token" con los datos del mismo.          */
/* ---------------------------------------------------------------------- */

int line_count = 1 ;
int current_file = 0 ;

struct _token token ;
struct _token token_prev ;
struct _token token_saved ;
static int use_saved = 0 ;

#define MAX_SOURCES 32

static const char * source_ptr ;
static const char * old_sources[MAX_SOURCES] ;
static int          old_line_counts[MAX_SOURCES] ;
static int          old_current_file[MAX_SOURCES] ;
static int          sources = 0 ;

void token_init (const char * source, int file)
{
	if (sources == MAX_SOURCES)
		compile_error (MSG_TOO_MANY_INCLUDES) ;

	old_line_counts [sources] = line_count ;
	old_current_file[sources] = current_file ;
	old_sources     [sources] = source_ptr ;
	sources++ ;

	line_count = 1 ;
	source_ptr = source ;
	if (!source_ptr)
	{
		fprintf (stdout, "token_init: no source\n") ;
		exit (1) ;
	}

        current_file = file ;
}

int token_endfile()
{
	if (sources > 0)
	{
		sources-- ;
		source_ptr = old_sources[sources] ;
		line_count = old_line_counts[sources] ;
                current_file = old_current_file[sources] ;
	}
	return 0 ;
}

void token_dump ()
{
	fprintf (stdout, "(") ;
	if (token.type == NUMBER) fprintf (stdout, "%d", token.code) ;
	else if (token.type == FLOAT) fprintf (stdout, "%f", token.value) ;
	else if (token.type == STRING) fprintf (stdout, "\"%s\"", string_get(token.code)) ;
	else if (token.type == IDENTIFIER) fprintf (stdout, "\"%s\"", identifier_name(token.code)) ;
	else if (token.type == NOTOKEN) fprintf (stdout, "EOF") ;
	else fprintf (stdout, "unknown\n") ;
	fprintf (stdout, ")\n") ;
}

extern void load_file (const char * filename) ;

void token_next ()
{
	static char buffer[1024] ;
	static int len ;
	char * buffer_ptr = buffer ;

	if (!source_ptr)
	{
		token.type = NOTOKEN ;
		return ;
	}

	if (use_saved)
	{
		token = token_saved ;
		use_saved = 0 ;
		return ;
	}
	token_prev = token ;

	while (*source_ptr)
	{
		while (ISSPACE(*source_ptr)) 
		{
			if (*source_ptr == '\n') line_count++ ;
			source_ptr++ ;
		}

		if (!*source_ptr)
		{
			token_endfile() ;
			if (source_ptr && *source_ptr) continue ;
			token.type = NOTOKEN ;
			return ;
		}
		
		/* Ignora comentarios */

		if (*source_ptr == '/' && *(source_ptr+1) == '*')
		{
			while (*source_ptr != '*' || source_ptr[1] != '/')
			{
				if (*source_ptr == '\n') line_count++ ;
				if (!*source_ptr)
				{
					token.type = NOTOKEN ;
					return ;
				}
				source_ptr++ ;
			}
			source_ptr += 2 ;
			continue ;
		}

		if (*source_ptr == '/' && *(source_ptr+1) == '/')
		{
			while (*source_ptr != '\n' && *source_ptr)
				source_ptr++ ;
			if (*source_ptr) source_ptr++ ;
			line_count++ ;
			continue ;
		}

		if (*source_ptr == '/' && *(source_ptr+1) == '*')
		{
			while (*source_ptr != '*' && *(source_ptr+1) != '/' && *source_ptr)
				source_ptr++ ;
			if (*source_ptr) source_ptr += 2 ;
			continue ;
		}

		/* Cadenas */

		if (*source_ptr == '"' || *source_ptr == '\'')
		{
			token.type = STRING ;
			token.code = string_compile(&source_ptr) ;
			return ;
		}

		/* Operadores de ms de un caracter */

		len = 0 ;

		if (*source_ptr == '<')
		{
			if (source_ptr[1] == '<')
			{
				if (source_ptr[2] == '=') len = 3 ; 
				else                      len = 2 ;
			}
			else if (source_ptr[1] == '>')	  len = 2 ;
			else if (source_ptr[1] == '=')	  len = 2 ;
			else                              len = 1 ;
		}
		else if (*source_ptr == '>')
		{
			if (source_ptr[1] == '>')
			{
				if (source_ptr[2] == '=') len = 3 ; 
				else                      len = 2 ;
			}
			else if (source_ptr[1] == '=')	  len = 2 ;
			else if (source_ptr[1] == '>')	  len = 2 ;
			else                              len = 1 ;
		}
		else if (*source_ptr == '|')
		{
			if (source_ptr[1] == '|')
			{
				if (source_ptr[2] == '=') len = 3 ; 
				else                      len = 2 ;
			}
			else if (source_ptr[1] == '=')    len = 2 ; 
			else                              len = 1 ;
		}
		else if (*source_ptr == '=')
		{
			if (source_ptr[1] == '=')	  len = 2 ;
			else if (source_ptr[1] == '>')	  len = 2 ;
			else if (source_ptr[1] == '<')	  len = 2 ;
			else                              len = 1 ;
		}
		else if (*source_ptr == '.')
		{
			if (source_ptr[1] == '.')	  len = 2 ;
			else                              len = 1 ;
		}
		else if (strchr("!&^%*+-/", *source_ptr))
		{
			if (source_ptr[1] == '=')	  len = 2 ;
			else if (strchr("+-&", *source_ptr) && 
			    source_ptr[1] == *source_ptr) len = 2 ;
			else                              len = 1 ;
		}

		if (len)
		{
			strncpy (buffer, source_ptr, len) ; 
			buffer[len] = 0 ;
			source_ptr += len ;
			token.code = identifier_search_or_add(buffer) ;
			token.type = IDENTIFIER ;
			return ;
		}
		
		/* Numbers */

		if (ISNUM(*source_ptr))
		{
			double num = 0, dec ;

			while (ISNUM(*source_ptr))
			{
				num = num * 10 + (*source_ptr++ - '0') ;
			}
			if ((*source_ptr) == '.')
			{
				source_ptr++ ;
				if (!ISNUM(*source_ptr))
					source_ptr-- ;
				else
				{
					dec = 0.1 ;
					while (ISNUM(*source_ptr))
					{
						num = num + dec * (*source_ptr++ - '0') ;
						dec /= 10.0 ;
					}
					token.type  = FLOAT ;
					token.value = (float)num ;
					return ;
				}
			}
			token.type = NUMBER ;
			token.code = (int)num ;
			token.value = (float)num ;
			return ;
		}

		/* Identificadores */

		if (ISWORDFIRST(*source_ptr))
		{
			while (ISWORDCHAR(*source_ptr))
			{
				if (buffer_ptr == buffer+1023)
					compile_error (MSG_IDENTIFIER_TOO_LONG) ;
				*buffer_ptr++ = TOUPPER(*source_ptr++) ;
			}
			*buffer_ptr++ = 0 ;
			token.code = identifier_search_or_add(buffer) ;
			token.type = IDENTIFIER ;

			/* Include */
			if (token.code == identifier_include && !use_saved)
			{
				while (ISSPACE(*source_ptr)) source_ptr++ ;
				if (*source_ptr == '"')
				{
					source_ptr++ ;
					buffer_ptr = buffer ;
					while (*source_ptr && *source_ptr != '"')
					{
						if (buffer_ptr == buffer+1023)
							compile_error (MSG_IDENTIFIER_TOO_LONG) ;
						*buffer_ptr++ = *source_ptr++ ;
					}
					if (*source_ptr == '"') source_ptr++ ;
					*buffer_ptr = 0 ;
					load_file (buffer) ;
					token_next() ;
					return ;
				}
				compile_error (MSG_IDENTIFIER_EXP) ;
			}
			return ;
		}

		/* 1-char operator or invalid symbol */

		if (!*source_ptr) break ;

		if (*source_ptr > 0 && *source_ptr < 32)
		{
			compile_error (MSG_INVALID_CHAR) ;
			return ;
		}

		*buffer_ptr++ = *source_ptr++ ;
		*buffer_ptr = 0 ;
		token.code = identifier_search_or_add(buffer) ;
		token.type = IDENTIFIER ;

		return ;
	}

	token.type = NOTOKEN ;
	return ;		/* End-of-file */
}

void token_back ()
{
	if (use_saved)
	{
		fprintf (stdout, "Error: two token_back in a row\n") ;
		exit(1) ;
	}

	token_saved = token ;
	token = token_prev ;
	use_saved = 1 ;
}
