/*
 *  Fenix - Videogame compiler/interpreter
 *  Current release       : FENIX - PROJECT 1.0 - R 0.76
 *  Last stable release   : 
 *  Project documentation : http://fenix.sourceforge.net
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  Copyright  1999 Jos Luis Cebrin Page
 *  Copyright  2002 Fenix Team
 *
 */

/*
 * FILE        : i_func.h
 * DESCRIPTION : Implements FENIX language function handlers
 *
 * HISTORY:  0.76 - Patched SAVE to real BYTE size on saving
 *			 0.76 - Corrected a clipper bug in MAP_BLOCK_COPY
 *			 0.76 - DRAW_AT patched to avoid XGRAPH rotation
 *			 0.75 - SET_ICON second WM fenix function to set window/taskbar icon
 *			 0.74 - SET_TITLE first WM fenix function to set window title
 *			 0.74 - WRITE_VAR, WRITE_INT, WRITE_FLOAT, WRITE_STRING
 *			 0.74 - Collision now accepts TYPE or PROCESS ID 
 *			 0.73 - Corrected advance & xadvance to init cos_tables
 *			 0.73 - Added some Win32 patches due to non-portable
 *					strftime formats
 *			 0.73 - Corrected div_set_center
 *			 0.73 - Added div_memory_free & div_memory_total
 *			 0.72 - Corrected div_get_real_point
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include <time.h>

#include "fxi.h"
#include "fmath.h"

/* WIN32 INCLUDES */
#ifndef LINUX
  #include <windows.h>
  #include <winbase.h>
#endif
/* LINUX INCLUDES */

#include "dcb.h"

/* Rutinas matemticas de punto fijo, basadas en Allegro */

fixed cos_table[90001] ;
int cos_table_initialized = 0 ;

void init_cos_tables()
{
	int i ;

	for (i = 0 ; i <= 90000 ; i++)
	{
		cos_table[i] = ftofix(cos (i * M_PI / 180000.0)) ;
	}
	cos_table_initialized = 1 ;
}

/* ---------------------------------------------------------------------- */
/* Funciones del sistema                                                  */
/* ---------------------------------------------------------------------- */

/* Funciones incompatibles con DIV */

static int div_say (INSTANCE * my, int * params)
{
    printf("%s\n", string_get(params[0])) ;
	string_discard(params[0]) ;
	return 1 ;
}

static int div_print (INSTANCE * my, int * params)
{

   WriteConsole (so, string_get(params[0]), lstrlen (string_get(params[0])),
                 &written, NULL);

   // Inicialization Console Functions

    string_discard(params[0]) ;
    return 1 ;
}

static int div_println (INSTANCE * my, int * params)
{
    printf("%s\n", string_get(params[0])) ;
	string_discard(params[0]) ;
    return 1 ;
}

static int div_printf (INSTANCE * my, int * params)
{
	return printf ("%f ", *(float *)&params[0]) ;
}

/* Funciones matemticas */

static int div_rand (INSTANCE * my, int * params)
{
	int num1 = params[0] ;
	int num2 = params[1] ;

	return num1 + (int)(((double)(num2-num1+1) * rand()) / (RAND_MAX+1.0)) ;
}

static int div_rand_seed (INSTANCE * my, int * params)
{
	srand (params[0]) ;
	return 1 ;
}

static int div_abs (INSTANCE * my, int * params)
{
	float num = *(float *)&params[0] ;
	float res = num < 0 ? -num:num ;
	return *(int *)&res ;
}

static int div_pow (INSTANCE * my, int * params)
{
	float res = (float)pow (*(float *)&params[0], *(float *)&params[1]) ;
	return *(int *)&res ;
}

static int div_fget_angle (INSTANCE * my, int * params)
{
	double dx = params[2] - params[0] ;
	double dy = params[3] - params[1] ;
	int angle ;

	if (dx == 0) return dy > 0 ? 270000 : 90000 ;

	angle = (int) (atan(dy / dx) * 180000.0 / M_PI) ;

	return dx > 0 ? -angle:-angle+180000 ;
}

static int div_fget_dist (INSTANCE * my, int * params)
{
	double dx = params[2] - params[0] ;
	double dy = params[3] - params[1] ;

	return (int)sqrt (dx*dx + dy*dy) ;
}

static int div_near_angle (INSTANCE * my, int * params)
{
	int angle = params[0] ;
	int dest  = params[1] ;
	int incr  = params[2] ;

	if (angle < dest && dest-angle > 180000)
		angle += 360000 ;

	if (angle > dest && angle-dest > 180000)
		angle -= 360000 ;

	if (angle < dest)
	{
		angle += incr ;
		if (angle > dest) angle = dest ;
	}
	else
	{
		angle -= incr ;
		if (angle < dest) angle = dest ;
	}

	if (angle < 0) return angle + 360000 ;
	if (angle >= 360000) return angle - 360000 ;
	return angle ;
}

static int div_sqrt (INSTANCE * my, int * params)
{
	float res = (float)sqrt(*(float *)&params[0]) ;
	return *(int *)&res ;
}

/* Funciones matemticas con floats */

static int div_cos (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)cos((double)(param*M_PI/180000.0)) ;
	return *(int *)&result ;
}
static int div_sin (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)sin((double)(param*M_PI/180000.0)) ;
	return *(int *)&result ;
}
static int div_tan (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)tan((double)(param*M_PI/180000.0)) ;
	return *(int *)&result ;
}
static int div_acos (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)(acos((double)param)*180000.0/M_PI) ;
	return *(int *)&result ;
}
static int div_asin (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)(asin((double)param)*180000.0/M_PI) ;
	return *(int *)&result ;
}
static int div_atan (INSTANCE * my, int * params)
{
	float param = *(float *)&params[0] ;
	float result = (float)(atan((double)param)*180000.0/M_PI) ;
	return *(int *)&result ;
}

/* Interaccin entre procesos */

static int div_exit (INSTANCE * my, int * params)
{
	INSTANCE * i = first_instance ;

	while (i)
	{
		LOCDWORD(i,STATUS) = STATUS_KILLED ;
		i = i->next ;
	}
	must_exit = 1 ;
	return must_exit ;
}

static int div_running (INSTANCE * my, int * params)
{
	INSTANCE * i = instance_get (params[0]) ;
	return i ? 1 : 0 ;
}

static int div_signal (INSTANCE * my, int * params)
{
	INSTANCE * i = instance_get (params[0]) ;
	int status ;
	int subcall[2] ;

	if (!params[0]) return 0 ;

	if (!i && params[0] < FIRST_INSTANCE_ID)
	{
		i = first_instance ;
		subcall[1] = params[1] ;
		while (i)
		{
			if (LOCDWORD(i,PROCESS_TYPE) == params[0])
			{
				subcall[0] = LOCDWORD(i, PROCESS_ID) ;
				div_signal (my, subcall) ;
			}
			i = i->next ;
		}
		return 0 ;
	}

	if (i)
	{
		switch (params[1] >= 100 ? params[1]-100:params[1])
		{
			case 0:		/* S_KILL */
				status = STATUS_KILLED ;
				LOCDWORD(i,STATUS) = status ;
				break ;

			case 1:		/* S_WAKEUP */
				status = STATUS_RUNNING ;
				LOCDWORD(i,STATUS) = status ;
				break ;

			case 2: 	/* S_SLEEP */
				status = STATUS_SLEEPING ;
				LOCDWORD(i,STATUS) = status ;
				break ;

			case 3:		/* S_FREEZE */
				status = STATUS_FROZEN ;
				LOCDWORD(i,STATUS) = status ;
				break ;

			default:
                printf ("Tipo de seal desconocida") ;
		}

		if (params[1] >= 100)
		{
			i = instance_getson(i) ;
			while (i)
			{
				subcall[0] = LOCDWORD(i,PROCESS_ID) ;
				subcall[1] = params[1] ;
				div_signal (my, subcall) ;
				i = instance_getbigbro(i)  ;
			}
		}
	}
	return 1 ;
}

static int div_let_me_alone (INSTANCE * my, int * params)
{
	INSTANCE * i = first_instance ;

	while (i)
	{
		if (i != my) LOCDWORD(i, STATUS) = STATUS_KILLED ;
		i = i->next ;
	}
	return 1 ;
}

static int div_get_disty (INSTANCE * my, int * params)
{
	double angle = params[0] * M_PI / 180000.0 ;
	return (int)(params[1] * -sin(angle)) ;
}

static int div_get_id (INSTANCE * my, int * params)
{
	INSTANCE * ptr = first_instance ;

	if (!params[0])
	{
		if (LOCDWORD(my, ID_SCAN))
		{
			ptr = instance_get (LOCDWORD(my,ID_SCAN)) ;
			if (ptr) ptr = ptr->next ;
		}
		while (ptr)
		{
			if (LOCDWORD(ptr,STATUS) >= STATUS_RUNNING) 
			{
				LOCDWORD(my,ID_SCAN) = LOCDWORD(ptr, PROCESS_ID) ;
				return LOCDWORD(ptr, PROCESS_ID) ;
			}
			ptr = ptr->next ;
		}
		LOCDWORD(my,ID_SCAN) = 0 ;
		return 0 ;
	}

	if (LOCDWORD(my,TYPE_SCAN))
	{
		ptr = instance_get (LOCDWORD(my,TYPE_SCAN)) ;
		if (LOCDWORD(ptr,PROCESS_TYPE) != params[0])
			ptr = first_instance ;
		else if (ptr)
			ptr = ptr->next ;
	}
	while (ptr)
	{
		if (LOCDWORD(ptr,PROCESS_TYPE) == params[0])
		{
			if (LOCDWORD(ptr,STATUS) >= STATUS_RUNNING) 
			{
				LOCDWORD(my,TYPE_SCAN) = LOCDWORD(ptr,PROCESS_ID);
				return LOCDWORD(ptr,PROCESS_ID) ;
			}
		}
		ptr = ptr->next ;
	}
	LOCDWORD(my,TYPE_SCAN) = 0 ;
	return 0 ;
}

/* Entrada/salida */

static int div_key (INSTANCE * my, int * params)
{
	int found = 0 ;
    /*
    key_equiv * curr ;
    curr = &key_table[params[0]] ;
	if (keytab_initialized==0) keytab_init() ;
	while (curr!=NULL && found==0) {		
		found = keystate[curr->sdlk_equiv] ;
		curr = curr->next ;
        }
    */
	return found ;
}

/* Funciones de acceso a ficheros */

static int div_save (INSTANCE * my, int * params)
{
	file * fp ;
	const char * filename ;
	int result ;

	filename = string_get (params[0]) ;
	if (!filename) return 0 ;

	fp = file_open (filename, "wb0") ;
	if (fp) 
	{
		result = file_write (fp, (void *)params[1], params[2]) ;
		file_close (fp) ;
	}
	string_discard (params[0]) ;
	return result ;
}

static int div_load (INSTANCE * my, int * params)
{
	file * fp ;
	const char * filename ;
	int result ;

	filename = string_get (params[0]) ;
	if (!filename) return 0 ;

	fp = file_open (filename, "rb0") ;
	if (!fp) return 0 ;

	result = file_read (fp, (void *)params[1], file_size(fp)) ;
	file_close (fp) ;
	string_discard(params[0]) ;

	return result ;
}

static int div_fopen (INSTANCE * my, int * params)
{
        static char * ops[] = { "rb0", "w+b0", "wb0", "rb", "wb6" } ;
        int r ;

        if (params[1] < 0 || params[1] > 4)
                params[0] = 0 ;

	r = (int) file_open (string_get(params[0]), ops[params[1]]) ;
	string_discard (params[0]) ;
	return r ;
}

static int div_fclose (INSTANCE * my, int * params)
{
	file_close ((file *)params[0]) ;
}

static int div_fread (INSTANCE * my, int * params)
{
	return file_read ((file *)params[0], (void *)params[1], params[2]) ;
}

static int div_fwrite (INSTANCE * my, int * params)
{
	return file_write ((file *)params[0], (void *)params[1], params[2]) ;
}

static int div_fseek (INSTANCE * my, int * params)
{
	return file_seek ((file *)params[0], params[1], params[2]) ;
}

static int div_ftell (INSTANCE * my, int * params)
{
	return file_pos ((file *)params[0]) ;
}

static int div_filelength (INSTANCE * my, int * params)
{
	return file_size ((file *)params[0]) ;
}

static int div_fputs (INSTANCE * my, int * params)
{
	int r = file_puts ((file *)params[0], string_get(params[1])) ;
	string_discard(params[1]) ;
	return r ;
}

static int div_file (INSTANCE * my, int * params)
{
	char buffer[1030], *ptr = buffer ;
        int str = string_new("") ;
        file * f ;

        f = file_open (string_get(params[0]), "rb") ;
        if (!f) return 0 ;
        string_discard(params[0]) ;
	
	while (!file_eof(f))
	{
		if (!file_read (f, ptr, 1))
                        break ;
                ptr++ ;
                if (ptr == buffer+1024) 
                {
                        *ptr = 0 ;
                        string_concat (str, buffer) ;
                        ptr = buffer ;
                }
	}
        *ptr = 0 ;
        string_concat (str, buffer) ;
	string_use (str) ;
        file_close (f) ;
	return str ;
}

static int div_fgets (INSTANCE * my, int * params)
{
	char buffer[1030] ;
	int str, str2 = 0, str3 ;
	int len, sigue ;
	
	for (;;)
	{
		file_gets ((file *)params[0], buffer, 1024) ;
		len = strlen(buffer) ;
		if (len > 1 && buffer[len-1] == '\n' && buffer[len-2] == '\\')
		{
			buffer[len-2] = 0 ;
			sigue = 1 ;
		}
		else	sigue = 0 ;

		str = string_new (buffer) ;
		if (str2)
		{
			str3 = string_add (str2, str) ;
			string_discard (str) ;
			string_discard (str2) ;
			str2 = str3 ;
		}
		else	str2 = str ;

		if (!sigue) break ;
	}
	string_use (str2) ;
	return str2 ;
}

static int div_feof (INSTANCE * my, int * params)
{
	return file_eof ((file *)params[0]) ;
}

/* Cadenas */

static int div_strlen (INSTANCE * my, int * params)
{
	const char * str = string_get(params[0]) ;
	int r = str ? strlen(str) : 0 ;
	string_discard (params[0]) ;
	return r ;
}

static int div_strupper (INSTANCE * my, int * params)
{
	int r = string_ucase(params[0]) ;
	string_discard (params[0]) ;
	string_use     (r) ;
	return r ;
}

static int div_strlower (INSTANCE * my, int * params)
{
	int r = string_lcase(params[0]) ;
	string_discard (params[0]) ;
	string_use     (r) ;
	return r ;
}

static int div_substr (INSTANCE * my, int * params)
{
	int r = string_substr(params[0],params[1],params[2]) ;
	string_discard (params[0]) ;
	string_use     (r) ;
	return r ;
}

static int div_strfind (INSTANCE * my, int * params)
{
	int r = string_find(params[0],params[1]) ;
	string_discard (params[0]) ;
	string_discard (params[1]) ;
	return r ;
}

static int div_itos (INSTANCE * my, int * params)
{
	int r = string_itoa (params[0]) ;
	string_use(r) ;
	return r ;
}

static int div_ftos (INSTANCE * my, int * params)
{
	int r = string_ftoa (*(float *)&params[0]) ;
	string_use(r) ;
	return r ;
}

static int div_stoi (INSTANCE * my, int * params)
{
	const char * str = string_get(params[0]) ;
	int r = str ? atoi(str) : 0 ;
	string_discard (params[0]) ;
	return r ;
}

static int div_stof (INSTANCE * my, int * params)
{
	const char * str = string_get(params[0]) ;
	float res = (float)(str ? atof(str) : 0 );
	string_discard (params[0]) ;
	return *(Sint32 *)&res ;
}

static int div_asc (INSTANCE * my, int * params)
{
	const Uint8 * str = string_get(params[0]) ;
	int r = str ? *str : 0 ;
	string_discard (params[0]) ;
	return r ;
}

static int div_chr (INSTANCE * my, int * params)
{
	char buffer[2] = " " ; int r ;
	buffer[0] = params[0] ;
	r = string_new (buffer) ;
	string_use (r) ;
	return r ;
}

/*
 * Dynamic memory
 */

#ifndef LINUX

/*
 *  WIN32 ONLY FUNCS
 */

/*
 *  FUNCTION : div_memory_free
 *
 *  Return number of free bytes (ONLY PHYSICAL MEMORY)
 *  NOTE: returned value is VOLATILE as mem is used by WM
 *  while fenix is running
 *
 *  PARAMS:
 *      no params
 *
 *  RETURN VALUE: 
 *      pointer to a float value...
 *
 */

static int div_memory_free (INSTANCE * my, int * params)
{
    MEMORYSTATUS mem ;
    float result ;

    GlobalMemoryStatus(&mem) ;
    result = (float)mem.dwAvailPhys ;
	
    return *(int *)&result ;
}

/*
 *  FUNCTION : div_memory_total
 *
 *  Return total number of bytes of memory (ONLY PHYSICAL MEMORY)
 *
 *  PARAMS:
 *      no params
 *
 *  RETURN VALUE: 
 *      pointer to a float value...
 *
 */

static int div_memory_total (INSTANCE * my, int * params)
{
    MEMORYSTATUS mem ;
    float result ;

    GlobalMemoryStatus(&mem) ;
    result = (float)mem.dwTotalPhys ;
	
    return *(int *)&result ;
}

/*
 * END OF WIN32 ONLY FUNCS
 */

#endif


static int div_memcopy (INSTANCE * my, int * params)
{
	memmove ((void *)params[0], (void *)params[1], params[2]) ;
}

static int div_memset (INSTANCE * my, int * params)
{
	memset ((void *)params[0], params[1], params[2]) ;
}

static int div_memsetw (INSTANCE * my, int * params)
{
	Uint16 * ptr = (Uint16 *)params[0] ;
	const Uint16 b = params[1] ;
	int n ;

	for (n = params[2] ; n ; n--) *ptr++ = b ;
}

static int div_alloc (INSTANCE * my, int * params)
{
	void * ptr = malloc (params[0]) ;
    if (!ptr) printf ("ALLOC: no hay memoria libre suficiente") ;
	return (int)ptr ;
}

static int div_realloc (INSTANCE * my, int * params)
{
	void * ptr = realloc ((void *)params[0], params[1]) ;
    if (!ptr) printf ("REALLOC: no hay memoria libre suficiente") ;
	return (int)ptr ;
}

static int div_free (INSTANCE * my, int * params)
{
	free ((void *)params[0]) ;
	return 1 ;
}

/* Hora del da */

static int div_time (INSTANCE * my, int * params)
{
	return time(0) ;
}

/*
 *  FUNCTION : div_ftime
 *
 *  Returns parts of the date
 *
 *  PARAMS:
 *      no params
 *
 *  RETURN VALUE: 
 *      pointer to a float value...
 *
 */

static int
div_ftime (INSTANCE * my, int * params)
{
	char buffer[128] ;
	char * format ;
	struct tm * t ;
	int ret ;
	time_t tim ;
	char * base ;
	
#ifndef LINUX
	/* aux buffer to make all changes... */
	char aux[128] ;
	unsigned char pos ;

#endif
	
	format = base = strdup(string_get(params[0])) ;
	
#ifndef LINUX
	/* Addapting win32 strftime formats to linux formats */
	/* HEAVY PATCH... :( */
	pos=0 ;
	while (*format && pos<127) {
		switch (*format) {
			case '%': /* MIGHT NEED CONVERSION... */
				aux[pos] = *format ;
				pos++ ;
				format++ ;
				switch (*format) {
					case 'e':	aux[pos] = '#' ;
								pos++ ;
								aux[pos] = 'd' ;								
								break ;
					case 'l':   aux[pos] = '#' ;
								pos++ ;
								aux[pos] = 'I' ;
								break ;
					case 'k':	aux[pos]='#' ;
								pos++ ;
								aux[pos] = 'H' ;								
								break ;
					case 'P':	aux[pos] = 'p' ;								
								break ;					
			
					case 'C':	aux[pos++] = '%' ;								
								aux[pos++] = *format ;
								aux[pos++] = '%' ;
								aux[pos] = 'Y' ;
								break ;
					
					case 'u':	aux[pos++] = '%' ;
								aux[pos++] = *format ;
								aux[pos++] = '%' ;
								aux[pos] = 'w' ;
								break ;
					
					case '%':	//MUST BE %%%% TO KEEP 2 IN POSTPROCESS
								aux[pos++] = '%' ;
								aux[pos++] = '%' ;
								aux[pos] = '%' ;
								break ;

					default:	aux[pos] = *format ;
								break ;
				}
				break ;
				
			default: aux[pos] = *format ;
					 break ;
			}
		format++ ;
		pos++ ;
	}
	aux[pos]=0 ;
	free(base) ;
	format = aux ;
#endif
	
	tim = (time_t) params[1] ;
	t = localtime(&tim) ;
	strftime (buffer, 128, format, t) ;
	string_discard(params[0]) ;

#ifndef LINUX
	/* win32 postprocess */
	aux[0] = '\0' ;
	format = buffer ;
	pos = 0 ;
	while (*format) {
		switch (*format) {
			case '%':	format++ ;
						switch (*format) {							
							case 'u':	format++ ;
										if (*format=='0') *format='7' ;
										aux[pos] = *format ;
										break ;

							case 'C':	format++ ;									
										aux[pos] = *format ;
										pos++ ;
										format++ ;										
										aux[pos] = *format ;										
										format++ ;										
										format++ ;										
										break ;

							default:	aux[pos] = *format ;
										break ;
						}
						break ;
			default:	aux[pos] = *format ;
						break ;
		}
		format++ ;
		pos++;	
	}
	aux[pos] = '\0' ;
	strcpy(buffer,aux) ;	
#endif

	ret = string_new(buffer) ;
	string_use(ret) ;
	return ret ;
}

/*
     Console functions
*/
static int div_cls (INSTANCE * my, int * params)
{

   COORD coord,coord_temp;
   DWORD written;

   coord.X = 0;
   for (coord.Y = 0; coord.Y <= 24; coord.Y++)
      FillConsoleOutputCharacter(so, ' ', 80, coord, &written);
   for (coord.Y = 0; coord.Y <= 79; coord.Y++)
      FillConsoleOutputAttribute(so, ( GLODWORD(TEXTCOLOR)-1 + ( 16 * ( GLODWORD(BACKCOLOR) - 1 ) )),
                                 80, coord, &written);
   coord_temp.X = 0; coord_temp.Y = 0;
   SetConsoleCursorPosition (so, coord_temp);

   return 1;
}

static int div_locate (INSTANCE * my, int * params)
{
    int x = params[0] ;
    int y = params[1] ;
    COORD coord_temp;

    coord_temp.X = x - 1; coord_temp.Y = y - 1;
    SetConsoleCursorPosition (so, coord_temp);

    return 1;
}

static int div_setcolor (INSTANCE * my, int * params)
{
    int texto = params[0] ;
    int fondo = params[1] ;

    SetConsoleTextAttribute (so, texto-1 + ( 16 * ( fondo - 1 ) ) );
    GLODWORD(TEXTCOLOR) = texto;
    GLODWORD(BACKCOLOR) = fondo;

    return 1;
}

static int div_getkey(INSTANCE * my, int * params)
{
  unsigned int c,ac;

  do {
     ReadConsoleInput(si, &inp, 1, (LPDWORD) &written);
     ac = inp.Event.KeyEvent.uChar.AsciiChar;
     if ( ac == 0 ) {
        ac = inp.Event.KeyEvent.wVirtualKeyCode;
     }
  } while ( ( ac == 0 ) || ( ac == 16 ) || ( ac == 20 ) || ( inp.Event.KeyEvent.bKeyDown == FALSE ) );

  return ac;
}

static int div_inkey(INSTANCE * my, int * params)
{
  unsigned int c,nRec;

  nRec =  PeekConsoleInput(si, &inp, 1, (LPDWORD) &written);

  if ( nRec ) {
     c = inp.Event.KeyEvent.wVirtualKeyCode;
     inp.Event.KeyEvent.wVirtualKeyCode = 0;
     FlushConsoleInputBuffer(si);
  } else
     c = 0;

  return c;
}


/*
 * Auxiliary QSort functions
 *
 */

double GetData(Uint8 *Data,int pos,int *params){
	int i; Uint8 b; Uint16 w; float f;
	if(params[4]==1){ return Data[pos*params[1]+params[3]];}
	if(params[4]==2){ memcpy(&w,&Data[pos*params[1]+params[3]],params[4]); return w;}
	if(params[4]==4 && params[5]==0){ memcpy(&i,&Data[pos*params[1]+params[3]],params[4]); return i;}
	if(params[4]==4 && params[5]==1){ memcpy(&f,&Data[pos*params[1]+params[3]],params[4]); return f;}
}

void QuickSort(Uint8 *Data,int inf, int sup, int *params)
{
	register izq,der;
	double mitad;
	Uint8* x=(Uint8*)malloc(params[1]);
	izq=inf;
	der=sup;
	mitad=GetData(Data,(izq+der)>>1,params);
	do
	{
		while(GetData(Data,izq,params) < mitad && izq<sup)
			izq++;
		while(mitad<GetData(Data,der,params) && der>inf)
			der--;

		if(izq<=der)
		{
			memcpy(x,&Data[izq*params[1]],params[1]);
			memcpy(&Data[izq*params[1]],&Data[der*params[1]],params[1]);
			memcpy(&Data[der*params[1]],x,params[1]);
			izq++;
			der--;
		}
	}while(izq<=der);
	
	if(inf<der)
		QuickSort(Data,inf,der,params);
	
	if(izq<sup)
		QuickSort(Data,izq,sup,params);
}		

/*
 *	QSort:
 */

static int div_quicksort(INSTANCE *my, int *params){ //punteroalarray,tamaodato,numdatos,offsetadatoordenador,tamaodatoaordenar,tipodato(int,float)
	Uint8 *Data=(Uint8 *)params[0];
	QuickSort(Data,0,params[2]-1,params);
}

/* ---------------------------------------------------------------------- */

#include "sysprocs.h"

static SYSPROC * sysproc_tab[256+MAX_SYSPROCS] ;

void sysproc_init()
{
	SYSPROC * proc = sysprocs ;

	while (proc->func)
	{
		sysproc_tab[proc->code] = proc ;
		proc++ ;
	}
}

int sysproc_add (char * name, char * paramtypes, int type, void * func)
{
	static SYSPROC * last = 0 ;
	static int sysproc_count = 0 ;

	if (!last) 
	{
		last = sysprocs ;
		sysproc_count++ ;
		while (last[1].func) last++, sysproc_count++ ;
	}
	last[1].code = last[0].code + 1 ;
	last[1].name = name ;
	last[1].paramtypes = paramtypes ;
	last[1].params = strlen(paramtypes) ;
	last[1].type = type ;
	last[1].func = func ;
	last[1].id   = 0 ;
	last++ ;
	sysproc_count++ ;
	if (sysproc_count == MAX_SYSPROCS)
        printf ("Demasiadas funciones del sistema") ;
	last[1].func = 0 ;
	return last->code ;
}

SYSPROC * sysproc_get (int code)
{
	return sysproc_tab[code] ;
}

//-----------------------------------------------
VOID setcursor (UINT shape) {
//-----------------------------------------------
   if (shape == nocursor)
      curinfo.bVisible = FALSE;
   else
      curinfo.bVisible = TRUE;
   if (shape == shortcursor) curinfo.dwSize = 20;
   if (shape == tallcursor) curinfo.dwSize = 90;
   if (shape == normalcursor) curinfo.dwSize = 5;
   SetConsoleCursorInfo (so, &curinfo);
}

